# This configuration file enables to find the image_dev package.
# If the detection of image_dev failed, the following variables are defined:
#   - image_dev_FOUND                -> False
#   - image_dev_INCLUDE_DIR          -> image_dev_INCLUDE_DIR-NOTFOUND
#   - image_dev_ALL_BINARIES_DEBUG   -> image_dev_ALL_BINARIES_DEBUG-NOTFOUND
#   - image_dev_ALL_BINARIES_RELEASE -> image_dev_ALL_BINARIES_RELEASE-NOTFOUND
# Else, the following variables are defined:
#   - image_dev_FOUND                -> True
#   - image_dev_INCLUDE_DIR          -> Path of the include directory containing the image_dev and IOLink header files.
#   - image_dev_ALL_BINARIES_DEBUG   -> List of all debug libraries that need to be installed next to the application.
#   - image_dev_ALL_BINARIES_RELEASE -> List of all release libraries that need to be installed next to the application.
#
# To link your project with image_dev, the following line is sufficient:
#   target_link_libraries( ${PROJECT_NAME} PUBLIC image_dev )
# This adds
#   - image_dev_INCLUDE_DIR in the list of include directiories of your project.
#   - A link to the image_dev and the IOLink libraries.
#
# To install all libraries used by image_dev, we must add the following lines:
#   install( FILES ${image_dev_ALL_BINARIES_DEBUG}   DESTINATION "your/output/debug/directory/path"   CONFIGURATIONS Debug )
#   install( FILES ${image_dev_ALL_BINARIES_RELEASE} DESTINATION "your/output/release/directory/path" CONFIGURATIONS Release )

# ---------------------------------------------------------------------------
# This macro exits of this file and writes the error message "error_message".
macro( exit_with_error error_message )
  if( image_dev_FIND_REQUIRED )
    message( FATAL_ERROR ${error_message} )
  elseif( NOT image_dev_FIND_QUIETLY )
    message( WARNING ${error_message} )
  endif()
  return()
endmacro()

# ---------------------------------------------------------------------------
# Initializes the variables of the image_dev package
set( image_dev_FOUND False )
set( image_dev_INCLUDE_DIR "image_dev_INCLUDE_DIR-NOTFOUND" )
set( image_dev_ALL_BINARIES_DEBUG "image_dev_ALL_BINARIES_DEBUG-NOTFOUND" )
set( image_dev_ALL_BINARIES_RELEASE "image_dev_ALL_BINARIES_RELEASE-NOTFOUND" )

# ---------------------------------------------------------------------------
# Checks the version of CMake
if( CMAKE_VERSION VERSION_LESS 3.7 )
  exit_with_error( "${CMAKE_FIND_PACKAGE_NAME} requires at least CMake version 3.7" )
endif()
# Checks that the OS is not APPLE
if( APPLE )
  exit_with_error( "${CMAKE_FIND_PACKAGE_NAME} does not available for APPLE" )
endif()

# ---------------------------------------------------------------------------
# Defines image_dev_HOME for convenience
get_filename_component( image_dev_HOME "${CMAKE_CURRENT_LIST_DIR}/.." ABSOLUTE )

# ---------------------------------------------------------------------------
# This defines necessary variables in order to find image_dev package
if( CMAKE_COMPILER_IS_GNUCXX )
  string( SUBSTRING ${CMAKE_CXX_COMPILER_VERSION} 0 1 major_version )
  string( SUBSTRING ${CMAKE_CXX_COMPILER_VERSION} 2 1 release_version )
  if( CMAKE_CXX_COMPILER_ID STREQUAL "GNU" )
    set( image_dev_CXX_COMPILER_ID "gcc" )
    if( CMAKE_CXX_COMPILER_VERSION VERSION_GREATER 5.0 )
      set( release_version "" ) # From GCC 5, only major version is significant.
    endif()
  endif()
  set( image_dev_CXX_COMPILER_VERSION "${major_version}${release_version}" CACHE STRING "GCC version used for prepacked" )
  # List of the supported versions (by binary compatibility).
  # This list will be used if we can't find the libraries directories for the current version of compiler.
  set( image_dev_CXX_COMPILER_OTHER_SUPPORTED_VERSION 9 7 )
else()
  string( TOLOWER ${CMAKE_CXX_COMPILER_ID} image_dev_CXX_COMPILER_ID )

  # determine the MSVC version from the Visual Studio version
  # and the list of the supported versions (by binary compatibility).
  # This list will be used if we can't find the libraries directories for the current version of Visual Studio.
  if ( MSVC_VERSION GREATER_EQUAL 1930 )
    set( image_dev_CXX_COMPILER_VERSION "17" )
    set( image_dev_CXX_COMPILER_OTHER_SUPPORTED_VERSION 17 16 15 14 )
  elseif ( MSVC_VERSION GREATER_EQUAL 1920 )
    set( image_dev_CXX_COMPILER_VERSION "16" )
    set( image_dev_CXX_COMPILER_OTHER_SUPPORTED_VERSION 16 15 14 )
  elseif ( MSVC_VERSION GREATER_EQUAL 1910 )
    set( image_dev_CXX_COMPILER_VERSION "15" )
    set( image_dev_CXX_COMPILER_OTHER_SUPPORTED_VERSION 15 14 )
  elseif ( MSVC_VERSION GREATER_EQUAL 1900 )
    set( image_dev_CXX_COMPILER_VERSION "14" )
    set( image_dev_CXX_COMPILER_OTHER_SUPPORTED_VERSION 14 )
  endif()
endif()

if( WIN32 )
  set( image_dev_SYSTEM_PROCESSOR x86_64)
else()
  set( image_dev_SYSTEM_PROCESSOR ${CMAKE_SYSTEM_PROCESSOR} )
endif()

set( image_dev_CXX_PLATFORM_ID ${CMAKE_CXX_PLATFORM_ID} )

set( CMAKE_ARCHITECTURE "${image_dev_CXX_PLATFORM_ID}-${image_dev_SYSTEM_PROCESSOR}-${image_dev_CXX_COMPILER_ID}${image_dev_CXX_COMPILER_VERSION}" )

# these variables are used by find_module_config to avoid rebuilding them for every module
set( image_dev_ARCH_DEBUG arch-${CMAKE_ARCHITECTURE}-debug )
set( image_dev_ARCH_RELEASE arch-${CMAKE_ARCHITECTURE}-release )

# If the "arch" directory does not exist, we check if another "arch" compatible directory exists
if( ( WIN32 OR ( CMAKE_BUILD_TYPE MATCHES "[dD]ebug" ) ) AND ( NOT EXISTS "${image_dev_HOME}/${image_dev_ARCH_DEBUG}" ) )
  foreach( cur_supported_version IN LISTS image_dev_CXX_COMPILER_OTHER_SUPPORTED_VERSION )
    set( new_debug_dir_path "arch-${image_dev_CXX_PLATFORM_ID}-${image_dev_SYSTEM_PROCESSOR}-${image_dev_CXX_COMPILER_ID}${cur_supported_version}-debug" )
    if ( EXISTS "${image_dev_HOME}/${new_debug_dir_path}" )
      set( image_dev_ARCH_DEBUG "${new_debug_dir_path}" )
      break()
    endif()
  endforeach()
endif()
if( ( WIN32 OR ( CMAKE_BUILD_TYPE MATCHES "[rR]elease" ) ) AND ( NOT EXISTS "${image_dev_HOME}/${image_dev_ARCH_RELEASE}" ) )
  foreach( cur_supported_version IN LISTS image_dev_CXX_COMPILER_OTHER_SUPPORTED_VERSION )
    set( new_release_dir_path "arch-${image_dev_CXX_PLATFORM_ID}-${image_dev_SYSTEM_PROCESSOR}-${image_dev_CXX_COMPILER_ID}${cur_supported_version}-release" )
    if ( EXISTS "${image_dev_HOME}/${new_release_dir_path}" )
      set( image_dev_ARCH_RELEASE "${new_release_dir_path}" )
      break()
    endif()
  endforeach()
endif()

# set image_dev_ARCH for convenience as it may be used in some places
if( CMAKE_BUILD_TYPE MATCHES "[dD]ebug" )
  set( image_dev_ARCH ${image_dev_ARCH_DEBUG} )
elseif( CMAKE_BUILD_TYPE MATCHES "[rR]elease" )
  set( image_dev_ARCH ${image_dev_ARCH_RELEASE} )
else()
  set( image_dev_ARCH arch-${CMAKE_ARCHITECTURE}-$<CONFIG> )
endif()


# ---------------------------------------------------------------------------
# Include directory
set( image_dev_INCLUDE_DIR "${image_dev_HOME}/include" )
if( NOT EXISTS "${image_dev_INCLUDE_DIR}/ImageDev/ImageDev.h" )
  set( image_dev_INCLUDE_DIR "image_dev_INCLUDE_DIR-NOTFOUND" )
  exit_with_error( "failed to find the include directory of image_dev" )
endif()
if( NOT EXISTS "${image_dev_INCLUDE_DIR}/iolink/IOLinkAPI.h" )
  set( image_dev_INCLUDE_DIR "image_dev_INCLUDE_DIR-NOTFOUND" )
  exit_with_error( "failed to find the include directory of IOLink" )
endif()

# Libraries directories
set( image_dev_LIBS_DIRECTORY_DEBUG "${image_dev_HOME}/${image_dev_ARCH_DEBUG}/lib/" )
set( image_dev_LIBS_DIRECTORY_RELEASE "${image_dev_HOME}/${image_dev_ARCH_RELEASE}/lib/" )

# Libraries path
if ( WIN32 )
  set( image_dev_LIBS_RELEASE "${image_dev_LIBS_DIRECTORY_RELEASE}imagedev_cpp.lib" )
  set( IOLink_LIBS_RELEASE "${image_dev_LIBS_DIRECTORY_RELEASE}iolink.lib" )
  set( image_dev_LIBS_DEBUG "${image_dev_LIBS_DIRECTORY_DEBUG}imagedev_cpp.lib" )
  set( IOLink_LIBS_DEBUG "${image_dev_LIBS_DIRECTORY_DEBUG}iolinkd.lib" )
else()
  set( image_dev_LIBS_RELEASE "${image_dev_LIBS_DIRECTORY_RELEASE}libimagedev_cpp.so" )
  set( IOLink_LIBS_RELEASE "${image_dev_LIBS_DIRECTORY_RELEASE}libiolink.so" )
  set( image_dev_LIBS_DEBUG "${image_dev_LIBS_DIRECTORY_DEBUG}libimagedev_cpp.so" )
  set( IOLink_LIBS_DEBUG "${image_dev_LIBS_DIRECTORY_DEBUG}libiolink.so" )
endif()

# Define the image_dev library
set( image_dev_LIBRARY_NAME image_dev )

add_library( ${image_dev_LIBRARY_NAME} INTERFACE IMPORTED GLOBAL )

set_property( TARGET ${image_dev_LIBRARY_NAME} PROPERTY INTERFACE_LINK_LIBRARIES ""
                               $<$<CONFIG:Release>:${image_dev_LIBS_RELEASE} ${IOLink_LIBS_RELEASE}>
                               $<$<CONFIG:RelWithDebInfo>:${image_dev_LIBS_RELEASE} ${IOLink_LIBS_RELEASE}>
                               $<$<CONFIG:MinSizeRel>:${image_dev_LIBS_RELEASE} ${IOLink_LIBS_RELEASE}>
                               $<$<CONFIG:Debug>:${image_dev_LIBS_DEBUG} ${IOLink_LIBS_DEBUG}> )

set_property( TARGET ${image_dev_LIBRARY_NAME} PROPERTY INTERFACE_INCLUDE_DIRECTORIES "${image_dev_INCLUDE_DIR}" )

# ---------------------------------------------------------------------------
# Define the list of all binaries

# Binaries directories
if ( WIN32 )
  set( image_dev_BINS_DIRECTORY_DEBUG "${image_dev_HOME}/${image_dev_ARCH_DEBUG}/bin/" )
  set( image_dev_BINS_DIRECTORY_RELEASE "${image_dev_HOME}/${image_dev_ARCH_RELEASE}/bin/" )
else()
  set( image_dev_BINS_DIRECTORY_DEBUG "${image_dev_LIBS_DIRECTORY_DEBUG}" )
  set( image_dev_BINS_DIRECTORY_RELEASE "${image_dev_LIBS_DIRECTORY_RELEASE}" )
endif()

# List of all binaries
if( WIN32 OR ( CMAKE_BUILD_TYPE MATCHES "[dD]ebug" ) )
  set( image_dev_ALL_BINARIES_DEBUG 
       "${image_dev_BINS_DIRECTORY_DEBUG}libiolink.so"
       "${image_dev_BINS_DIRECTORY_DEBUG}libiolink_cuda.so"
       "${image_dev_BINS_DIRECTORY_DEBUG}libonnxruntime.so"
       "${image_dev_BINS_DIRECTORY_DEBUG}libonnxruntime.so.1.15.1"
       "${image_dev_BINS_DIRECTORY_DEBUG}libonnxruntime_providers_cuda.so"
       "${image_dev_BINS_DIRECTORY_DEBUG}libonnxruntime_providers_shared.so"
       "${image_dev_BINS_DIRECTORY_DEBUG}libimagedev_tbb.so"
       "${image_dev_BINS_DIRECTORY_DEBUG}libiomp5.so"
       "${image_dev_BINS_DIRECTORY_DEBUG}libimagedev_proxy_fnp.so"
       "${image_dev_BINS_DIRECTORY_DEBUG}libimagedev_object_model.so"
       "${image_dev_BINS_DIRECTORY_DEBUG}libimagedev_vip.so"
       "${image_dev_BINS_DIRECTORY_DEBUG}libimagedev_vip_magn.so"
       "${image_dev_BINS_DIRECTORY_DEBUG}libimagedev_vip_jero.so"
       "${image_dev_BINS_DIRECTORY_DEBUG}libimagedev_vip_nabu.so"
       "${image_dev_BINS_DIRECTORY_DEBUG}libimagedev_vip_onnx.so"
       "${image_dev_BINS_DIRECTORY_DEBUG}libimagedev_vip_cuda.so"
       "${image_dev_BINS_DIRECTORY_DEBUG}libimagedev_util.so"
       "${image_dev_BINS_DIRECTORY_DEBUG}libimagedev_documentation.so"
       "${image_dev_BINS_DIRECTORY_DEBUG}libimagedev_inside_length.so"
       "${image_dev_BINS_DIRECTORY_DEBUG}libimagedev_visilib.so"
       "${image_dev_BINS_DIRECTORY_DEBUG}libimagedev_porosity.so"
       "${image_dev_BINS_DIRECTORY_DEBUG}libimagedev_service_handler.so"
       "${image_dev_BINS_DIRECTORY_DEBUG}libimagedev_access_visilib.so"
       "${image_dev_BINS_DIRECTORY_DEBUG}libimagedev_data.so"
       "${image_dev_BINS_DIRECTORY_DEBUG}libimagedev_process.so"
       "${image_dev_BINS_DIRECTORY_DEBUG}libimagedev_visilog_api.so"
       "${image_dev_BINS_DIRECTORY_DEBUG}libimagedev_proxy.so"
       "${image_dev_BINS_DIRECTORY_DEBUG}libimagedev_cpp.so" )
endif()
if( WIN32 OR ( CMAKE_BUILD_TYPE MATCHES "[rR]elease" ) )
  set( image_dev_ALL_BINARIES_RELEASE 
       "${image_dev_BINS_DIRECTORY_RELEASE}libiolink.so"
       "${image_dev_BINS_DIRECTORY_RELEASE}libiolink_cuda.so"
       "${image_dev_BINS_DIRECTORY_RELEASE}libonnxruntime.so"
       "${image_dev_BINS_DIRECTORY_RELEASE}libonnxruntime.so.1.15.1"
       "${image_dev_BINS_DIRECTORY_RELEASE}libonnxruntime_providers_cuda.so"
       "${image_dev_BINS_DIRECTORY_RELEASE}libonnxruntime_providers_shared.so"
       "${image_dev_BINS_DIRECTORY_RELEASE}libimagedev_tbb.so"
       "${image_dev_BINS_DIRECTORY_RELEASE}libiomp5.so"
       "${image_dev_BINS_DIRECTORY_RELEASE}libimagedev_proxy_fnp.so"
       "${image_dev_BINS_DIRECTORY_RELEASE}libimagedev_object_model.so"
       "${image_dev_BINS_DIRECTORY_RELEASE}libimagedev_vip.so"
       "${image_dev_BINS_DIRECTORY_RELEASE}libimagedev_vip_magn.so"
       "${image_dev_BINS_DIRECTORY_RELEASE}libimagedev_vip_jero.so"
       "${image_dev_BINS_DIRECTORY_RELEASE}libimagedev_vip_nabu.so"
       "${image_dev_BINS_DIRECTORY_RELEASE}libimagedev_vip_onnx.so"
       "${image_dev_BINS_DIRECTORY_RELEASE}libimagedev_vip_cuda.so"
       "${image_dev_BINS_DIRECTORY_RELEASE}libimagedev_util.so"
       "${image_dev_BINS_DIRECTORY_RELEASE}libimagedev_documentation.so"
       "${image_dev_BINS_DIRECTORY_RELEASE}libimagedev_inside_length.so"
       "${image_dev_BINS_DIRECTORY_RELEASE}libimagedev_visilib.so"
       "${image_dev_BINS_DIRECTORY_RELEASE}libimagedev_porosity.so"
       "${image_dev_BINS_DIRECTORY_RELEASE}libimagedev_service_handler.so"
       "${image_dev_BINS_DIRECTORY_RELEASE}libimagedev_access_visilib.so"
       "${image_dev_BINS_DIRECTORY_RELEASE}libimagedev_data.so"
       "${image_dev_BINS_DIRECTORY_RELEASE}libimagedev_process.so"
       "${image_dev_BINS_DIRECTORY_RELEASE}libimagedev_visilog_api.so"
       "${image_dev_BINS_DIRECTORY_RELEASE}libimagedev_proxy.so"
       "${image_dev_BINS_DIRECTORY_RELEASE}libimagedev_cpp.so" )
endif()

# The image_dev package was found.
set( image_dev_FOUND True )
