/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/ImageDevCppExports.h>
#include <memory>

namespace imagedev {

class MorphologicalKernel3dPrivate;

/**
 * This class represents a 3D morphological Kernel.
 *
 * A 3D morphological kernel contains exactly 27 values and indexes matching the following table :
 * 0 1 2            9 10 11           18 19 20
 * 3 4 5 for z=0,  12 13 14 for z=1,  21 22 23  for z=2
 * 6 7 8           15 16 17           24 25 26
 *
 * Accepted values for the kernel are Value :
 * - ZERO: pixel is off
 * - ONE: pixel is on
 * - ANY: pixel is ignored
 */
class IMAGEDEV_CPP_API MorphologicalKernel3d
{
public:
    // possible values
    enum Value
    {
        ZERO = 0,
        ONE = 1,
        ANY = 2
    };

    typedef MorphologicalKernel3d Self;
    typedef std::shared_ptr< Self > Ptr;

    /** constructor returning a shared ptr with all kernel's element initialized with the same value
     * @param value common value for all the kernel's elements.
     */
    static Ptr New( Value value = ZERO );

    /** Get a value of a specific kernel's element
     * @param index position of the kernel's element.
     */
    Value getValue( int index );

    /** Affect a value to a specific kernel's element
     * @param index position of the kernel's element.
     * @param value value to affect.
     */
    void setValue( int index, Value value );

    /** Affect all kernel's elements with the same value
     * @param value default value for all kernel's elements.
     */
    void setAllValues( Value value = ZERO );

    /** Direct rotation on X axis of 2 PI / 8
     * View on first slice on X
     *   18   9  0         9   0   3
     *   21  12  3   =>   18  12   6
     *   24  15  6        21  24  15
     * @param rotation number of rotation of 2 PI / 8.
     */
    void rotateX( size_t rotation = 1 );

    /** Direct rotation on Y axis of 2 PI / 8
     * View on first slice on Y
     *    6   7   8         7   8  17
     *   15  16  17   =>    6  16  26
     *   24  25  26        15  24  25
     * @param rotation number of rotation of 2 PI / 8.
     */
    void rotateY( size_t rotation = 1 );

    /** Direct rotation on Z axis of 2 PI / 8
     * View on first slice on Z
     *    0   1   2         1   2   5
     *    3   4   5   =>    0   4   8
     *    6   7   8         3   6   7
     * @param rotation number of rotation of 2 PI / 8.
     */
    void rotateZ( size_t rotation = 1 );

    // destructor
    ~MorphologicalKernel3d();

    friend class MorphologicalKernel3dPrivate;
    friend class MorphologicalLut3dPrivate;

protected:
    MorphologicalKernel3d( Value value = ZERO );

    MorphologicalKernel3dPrivate* m_p;
};

} // namespace imagedev
