/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <iolink/view/ImageView.h>

namespace imagedev
{
/// Performs a binarization of a grayscale image based on the mean intensity of a sliding window centered around each voxel.
class IMAGEDEV_CPP_API AdaptiveThresholding3d final : public GenericAlgorithm
{
public:
    /// The comparison test to perform between image and value.
    enum ComparisonCriterion
    {
    /// All voxels whose value is greater than or equal to the local threshold value are set to 1, others are set to 0.
        GREATER_OR_EQUAL = 0,
    /// All voxels whose value is less than or equal to the local threshold value are set to 1, others are set to 0.
        LESS_OR_EQUAL
    };
    /// The local thresholding mode.
    enum ThresholdMode
    {
    /// The local threshold is equal to mean(I) x threshold.
        MULTIPLICATIVE = 0,
    /// The local threshold is equal to mean(I) + threshold.
        ADDITIVE
    };

    // Command constructor.
    AdaptiveThresholding3d();


    /// Gets the inputImage parameter.
    /// The input image. Its type can be integer or float.
    std::shared_ptr< iolink::ImageView > inputImage() const;
    /// Sets the inputImage parameter.
    /// The input image. Its type can be integer or float.
    void setInputImage( std::shared_ptr< iolink::ImageView > inputImage );

    /// Gets the kernelRadiusX parameter.
    /// The half horizontal size of the kernel size in pixels. A cube structuring element always has an odd side length (3x3x3, 5x5x5, etc.) which is defined by twice the kernel radius + 1.
    int32_t kernelRadiusX() const;
    /// Sets the kernelRadiusX parameter.
    /// The half horizontal size of the kernel size in pixels. A cube structuring element always has an odd side length (3x3x3, 5x5x5, etc.) which is defined by twice the kernel radius + 1.
    void setKernelRadiusX( const int32_t& kernelRadiusX );

    /// Gets the kernelRadiusY parameter.
    /// The half vertical size of the kernel size in pixels. A cube structuring element always has an odd side length (3x3x3, 5x5x5, etc.) which is defined by twice the kernel radius + 1.
    int32_t kernelRadiusY() const;
    /// Sets the kernelRadiusY parameter.
    /// The half vertical size of the kernel size in pixels. A cube structuring element always has an odd side length (3x3x3, 5x5x5, etc.) which is defined by twice the kernel radius + 1.
    void setKernelRadiusY( const int32_t& kernelRadiusY );

    /// Gets the kernelRadiusZ parameter.
    /// The half depth size of the kernel size in pixels. A cube structuring element always has an odd side length (3x3x3, 5x5x5, etc.) which is defined by twice the kernel radius + 1.
    int32_t kernelRadiusZ() const;
    /// Sets the kernelRadiusZ parameter.
    /// The half depth size of the kernel size in pixels. A cube structuring element always has an odd side length (3x3x3, 5x5x5, etc.) which is defined by twice the kernel radius + 1.
    void setKernelRadiusZ( const int32_t& kernelRadiusZ );

    /// Gets the threshold parameter.
    /// The fraction or the additive thresholding value, according to the thresholdMode parameter value.
    double threshold() const;
    /// Sets the threshold parameter.
    /// The fraction or the additive thresholding value, according to the thresholdMode parameter value.
    void setThreshold( const double& threshold );

    /// Gets the comparisonCriterion parameter.
    /// The comparison test to perform between image and value.
    AdaptiveThresholding3d::ComparisonCriterion comparisonCriterion() const;
    /// Sets the comparisonCriterion parameter.
    /// The comparison test to perform between image and value.
    void setComparisonCriterion( const AdaptiveThresholding3d::ComparisonCriterion& comparisonCriterion );

    /// Gets the thresholdMode parameter.
    /// The local thresholding mode.
    AdaptiveThresholding3d::ThresholdMode thresholdMode() const;
    /// Sets the thresholdMode parameter.
    /// The local thresholding mode.
    void setThresholdMode( const AdaptiveThresholding3d::ThresholdMode& thresholdMode );

    /// Gets the outputBinaryImage parameter.
    /// The output binary image. Its dimensions are forced to the same values as the input.
    std::shared_ptr< iolink::ImageView > outputBinaryImage() const;
    /// Sets the outputBinaryImage parameter.
    /// The output binary image. Its dimensions are forced to the same values as the input.
    void setOutputBinaryImage( std::shared_ptr< iolink::ImageView > outputBinaryImage );

    // Method to launch the command.
    void execute();

};

/// Performs a binarization of a grayscale image based on the mean intensity of a sliding window centered around each voxel.
/// @param inputImage The input image. Its type can be integer or float.
/// @param kernelRadiusX The half horizontal size of the kernel size in pixels. A cube structuring element always has an odd side length (3x3x3, 5x5x5, etc.) which is defined by twice the kernel radius + 1.
/// @param kernelRadiusY The half vertical size of the kernel size in pixels. A cube structuring element always has an odd side length (3x3x3, 5x5x5, etc.) which is defined by twice the kernel radius + 1.
/// @param kernelRadiusZ The half depth size of the kernel size in pixels. A cube structuring element always has an odd side length (3x3x3, 5x5x5, etc.) which is defined by twice the kernel radius + 1.
/// @param threshold The fraction or the additive thresholding value, according to the thresholdMode parameter value.
/// @param comparisonCriterion The comparison test to perform between image and value.
/// @param thresholdMode The local thresholding mode.
/// @param outputBinaryImage The output binary image. Its dimensions are forced to the same values as the input.
/// @return Returns the outputBinaryImage output parameter.
IMAGEDEV_CPP_API 
std::shared_ptr< iolink::ImageView >
adaptiveThresholding3d( std::shared_ptr< iolink::ImageView > inputImage,
                        int32_t kernelRadiusX,
                        int32_t kernelRadiusY,
                        int32_t kernelRadiusZ,
                        double threshold,
                        AdaptiveThresholding3d::ComparisonCriterion comparisonCriterion,
                        AdaptiveThresholding3d::ThresholdMode thresholdMode,
                        std::shared_ptr< iolink::ImageView > outputBinaryImage = nullptr );
} // namespace imagedev
