/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <iolink/Vector.h>
#include <iolink/view/ImageView.h>

namespace imagedev
{
/// Creates a new two-dimensional image containing a binary or labeled grid.
class IMAGEDEV_CPP_API BoxGridImage2d final : public GenericAlgorithm
{
public:
    /// OutputType defines the type of desired grid in the output image.
    enum OutputType
    {
    /// The output image contains a binary grid.
        LINES = 0,
    /// The output image contains a grid of connected labels.
        CONTIGUOUS_REGIONS,
    /// The output image contains a grid of separated labels.
        SEPARATED_REGIONS
    };

    // Command constructor.
    BoxGridImage2d();


    /// Gets the outputType parameter.
    /// OutputType defines the type of desired grid in the output image.
    BoxGridImage2d::OutputType outputType() const;
    /// Sets the outputType parameter.
    /// OutputType defines the type of desired grid in the output image.
    void setOutputType( const BoxGridImage2d::OutputType& outputType );

    /// Gets the imageSize parameter.
    /// The dimensions in pixels of the output image.
    iolink::Vector2u32 imageSize() const;
    /// Sets the imageSize parameter.
    /// The dimensions in pixels of the output image.
    void setImageSize( const iolink::Vector2u32& imageSize );

    /// Gets the origin parameter.
    /// The origin of the first full rectangular region to generate.
    iolink::Vector2u32 origin() const;
    /// Sets the origin parameter.
    /// The origin of the first full rectangular region to generate.
    void setOrigin( const iolink::Vector2u32& origin );

    /// Gets the gridOffset parameter.
    /// The X and Y side size of boxes to create in the output images. In CONTIGUOUS_REGIONS or SEPARATED_REGIONS mode, it represents the number of pixels between the top left corners of contiguous labels in each direction. In LINES mode, it represents the offset, in pixels, between consecutive lines in each direction.
    iolink::Vector2u32 gridOffset() const;
    /// Sets the gridOffset parameter.
    /// The X and Y side size of boxes to create in the output images. In CONTIGUOUS_REGIONS or SEPARATED_REGIONS mode, it represents the number of pixels between the top left corners of contiguous labels in each direction. In LINES mode, it represents the offset, in pixels, between consecutive lines in each direction.
    void setGridOffset( const iolink::Vector2u32& gridOffset );

    /// Gets the outputImage parameter.
    /// The output 2D image containing the grid. Type depends of the type of the grid.
    std::shared_ptr< iolink::ImageView > outputImage() const;
    /// Sets the outputImage parameter.
    /// The output 2D image containing the grid. Type depends of the type of the grid.
    void setOutputImage( std::shared_ptr< iolink::ImageView > outputImage );

    // Method to launch the command.
    void execute();

};

/// Creates a new two-dimensional image containing a binary or labeled grid.
/// @param outputType OutputType defines the type of desired grid in the output image.
/// @param imageSize The dimensions in pixels of the output image.
/// @param origin The origin of the first full rectangular region to generate.
/// @param gridOffset The X and Y side size of boxes to create in the output images. In CONTIGUOUS_REGIONS or SEPARATED_REGIONS mode, it represents the number of pixels between the top left corners of contiguous labels in each direction. In LINES mode, it represents the offset, in pixels, between consecutive lines in each direction.
/// @param outputImage The output 2D image containing the grid. Type depends of the type of the grid.
/// @return Returns the outputImage output parameter.
IMAGEDEV_CPP_API 
std::shared_ptr< iolink::ImageView >
boxGridImage2d( BoxGridImage2d::OutputType outputType,
                const iolink::Vector2u32& imageSize,
                const iolink::Vector2u32& origin,
                const iolink::Vector2u32& gridOffset,
                std::shared_ptr< iolink::ImageView > outputImage = nullptr );
} // namespace imagedev
