/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <iolink/ArrayX.h>
#include <memory>
#include <iolink/view/ImageView.h>

namespace imagedev
{
// Output structure of the centerline3d function.
struct Centerline3dOutput
{
    /// The output image. Its size and type are forced to the same values as the input.
    std::shared_ptr< iolink::ImageView > outputObjectImage;
    /// The list of indices of the output line set.
    std::shared_ptr<iolink::ArrayXi64> outputIndices;
    /// The list of vertices of the output line set.
    std::shared_ptr<iolink::ArrayXd> outputVertices;
};

/// Extracts the centerline of a binary three-dimensional image.
class IMAGEDEV_CPP_API Centerline3d final : public GenericAlgorithm
{
public:

    // Command constructor.
    Centerline3d();


    /// Gets the inputObjectImage parameter.
    /// The binary or label input image.
    std::shared_ptr< iolink::ImageView > inputObjectImage() const;
    /// Sets the inputObjectImage parameter.
    /// The binary or label input image.
    void setInputObjectImage( std::shared_ptr< iolink::ImageView > inputObjectImage );

    /// Gets the autoMode parameter.
    /// The way to determine the number of branches.
    bool autoMode() const;
    /// Sets the autoMode parameter.
    /// The way to determine the number of branches.
    void setAutoMode( const bool& autoMode );

    /// Gets the numberOfBranches parameter.
    /// The expected number of branches. This parameter is ignored if autoMode is set to YES.
    uint32_t numberOfBranches() const;
    /// Sets the numberOfBranches parameter.
    /// The expected number of branches. This parameter is ignored if autoMode is set to YES.
    void setNumberOfBranches( const uint32_t& numberOfBranches );

    /// Gets the newBranchSensibility parameter.
    /// The minimum length of a branch added automatically. This parameter is ignored if autoMode is set to NO.
    double newBranchSensibility() const;
    /// Sets the newBranchSensibility parameter.
    /// The minimum length of a branch added automatically. This parameter is ignored if autoMode is set to NO.
    void setNewBranchSensibility( const double& newBranchSensibility );

    /// Gets the distanceMapSmoothing parameter.
    /// The smoothing level of the distance map. It controls the standard deviation of a Gaussian filter on the distance map, which is computed internally to make the tree centered. Smoothing the distance map can remove artifacts in the tree, like oscillations.
    double distanceMapSmoothing() const;
    /// Sets the distanceMapSmoothing parameter.
    /// The smoothing level of the distance map. It controls the standard deviation of a Gaussian filter on the distance map, which is computed internally to make the tree centered. Smoothing the distance map can remove artifacts in the tree, like oscillations.
    void setDistanceMapSmoothing( const double& distanceMapSmoothing );

    /// Gets the lineSetSmoothing parameter.
    /// The number of smoothing iterations applied to the output polyline set. Each smoothing iteration applies a mean filter of size 3 on the vertices of the polylines.
    uint32_t lineSetSmoothing() const;
    /// Sets the lineSetSmoothing parameter.
    /// The number of smoothing iterations applied to the output polyline set. Each smoothing iteration applies a mean filter of size 3 on the vertices of the polylines.
    void setLineSetSmoothing( const uint32_t& lineSetSmoothing );

    /// Gets the outputIndices parameter.
    /// The list of indices of the output line set.
    std::shared_ptr<iolink::ArrayXi64> outputIndices() const;
    /// Sets the outputIndices parameter.
    /// The list of indices of the output line set.
    void setOutputIndices( std::shared_ptr<iolink::ArrayXi64> outputIndices );

    /// Gets the outputVertices parameter.
    /// The list of vertices of the output line set.
    std::shared_ptr<iolink::ArrayXd> outputVertices() const;
    /// Sets the outputVertices parameter.
    /// The list of vertices of the output line set.
    void setOutputVertices( std::shared_ptr<iolink::ArrayXd> outputVertices );

    /// Gets the outputObjectImage parameter.
    /// The output image. Its size and type are forced to the same values as the input.
    std::shared_ptr< iolink::ImageView > outputObjectImage() const;
    /// Sets the outputObjectImage parameter.
    /// The output image. Its size and type are forced to the same values as the input.
    void setOutputObjectImage( std::shared_ptr< iolink::ImageView > outputObjectImage );

    // Method to launch the command.
    void execute();

};

/// Extracts the centerline of a binary three-dimensional image.
/// @param inputObjectImage The binary or label input image.
/// @param autoMode The way to determine the number of branches.
/// @param numberOfBranches The expected number of branches. This parameter is ignored if autoMode is set to YES.
/// @param newBranchSensibility The minimum length of a branch added automatically. This parameter is ignored if autoMode is set to NO.
/// @param distanceMapSmoothing The smoothing level of the distance map. It controls the standard deviation of a Gaussian filter on the distance map, which is computed internally to make the tree centered. Smoothing the distance map can remove artifacts in the tree, like oscillations.
/// @param lineSetSmoothing The number of smoothing iterations applied to the output polyline set. Each smoothing iteration applies a mean filter of size 3 on the vertices of the polylines.
/// @param outputIndices The list of indices of the output line set.
/// @param outputVertices The list of vertices of the output line set.
/// @param outputObjectImage The output image. Its size and type are forced to the same values as the input.
/// @return Returns a Centerline3dOutput structure containing the outputIndices, outputVertices and outputObjectImage output parameters.
IMAGEDEV_CPP_API 
Centerline3dOutput
centerline3d( std::shared_ptr< iolink::ImageView > inputObjectImage,
              bool autoMode,
              uint32_t numberOfBranches,
              double newBranchSensibility,
              double distanceMapSmoothing,
              uint32_t lineSetSmoothing,
              std::shared_ptr<iolink::ArrayXi64> outputIndices = nullptr,
              std::shared_ptr<iolink::ArrayXd> outputVertices = nullptr,
              std::shared_ptr< iolink::ImageView > outputObjectImage = nullptr );
} // namespace imagedev
