/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <iolink/Vector.h>
#include <iolink/view/ImageView.h>

namespace imagedev
{
/// Performs a contrast limited adaptive histogram equalization (CLAHE) on a three-dimensional image.
class IMAGEDEV_CPP_API Clahe3d final : public GenericAlgorithm
{
public:
    /// The way to compute histograms for the transformation function.
    enum ComputationMode
    {
    /// The image is divided in contiguous tiles. A histogram is calculated for each tile and the transformation function for a given pixel is determined by interpolation of these histograms.
        INTERPOLATED = 0,
    /// A histogram is computed for the tile surrounding each pixel and the transformation function is based on this histogram.
        SLIDING
    };

    // Command constructor.
    Clahe3d();


    /// Gets the inputImage parameter.
    /// The input image.
    std::shared_ptr< iolink::ImageView > inputImage() const;
    /// Sets the inputImage parameter.
    /// The input image.
    void setInputImage( std::shared_ptr< iolink::ImageView > inputImage );

    /// Gets the tileSize parameter.
    /// The X, Y and Z size, in pixels, of the tile used to compute the histogram.
    iolink::Vector3u32 tileSize() const;
    /// Sets the tileSize parameter.
    /// The X, Y and Z size, in pixels, of the tile used to compute the histogram.
    void setTileSize( const iolink::Vector3u32& tileSize );

    /// Gets the computationMode parameter.
    /// The way to compute histograms for the transformation function.
    Clahe3d::ComputationMode computationMode() const;
    /// Sets the computationMode parameter.
    /// The way to compute histograms for the transformation function.
    void setComputationMode( const Clahe3d::ComputationMode& computationMode );

    /// Gets the clipLimit parameter.
    /// The ratio defining the maximum number of elements contained in a histogram bin as a multiple of the average histogram contents.
    double clipLimit() const;
    /// Sets the clipLimit parameter.
    /// The ratio defining the maximum number of elements contained in a histogram bin as a multiple of the average histogram contents.
    void setClipLimit( const double& clipLimit );

    /// Gets the numberOfBins parameter.
    /// The number of bins of the histogram.
    uint32_t numberOfBins() const;
    /// Sets the numberOfBins parameter.
    /// The number of bins of the histogram.
    void setNumberOfBins( const uint32_t& numberOfBins );

    /// Gets the outputImage parameter.
    /// The output image. The output image characteristics are forced to the same as the input image.
    std::shared_ptr< iolink::ImageView > outputImage() const;
    /// Sets the outputImage parameter.
    /// The output image. The output image characteristics are forced to the same as the input image.
    void setOutputImage( std::shared_ptr< iolink::ImageView > outputImage );

    // Method to launch the command.
    void execute();

};

/// Performs a contrast limited adaptive histogram equalization (CLAHE) on a three-dimensional image.
/// @param inputImage The input image.
/// @param tileSize The X, Y and Z size, in pixels, of the tile used to compute the histogram.
/// @param computationMode The way to compute histograms for the transformation function.
/// @param clipLimit The ratio defining the maximum number of elements contained in a histogram bin as a multiple of the average histogram contents.
/// @param numberOfBins The number of bins of the histogram.
/// @param outputImage The output image. The output image characteristics are forced to the same as the input image.
/// @return Returns the outputImage output parameter.
IMAGEDEV_CPP_API 
std::shared_ptr< iolink::ImageView >
clahe3d( std::shared_ptr< iolink::ImageView > inputImage,
         const iolink::Vector3u32& tileSize,
         Clahe3d::ComputationMode computationMode,
         double clipLimit,
         uint32_t numberOfBins,
         std::shared_ptr< iolink::ImageView > outputImage = nullptr );
} // namespace imagedev
