/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <iolink/Vector.h>
#include <iolink/view/DataFrameView.h>
#include <ImageDev/Data/Cuda/CudaContext.h>
#include <iolink/view/ImageView.h>

namespace imagedev
{
// Output structure of the cudaTemplateMatching3d function.
struct CudaTemplateMatching3dOutput
{
    /// The correlation scores image. Its dimensions are forced to the same values as the input. Its data type is forced to floating point.
    std::shared_ptr< iolink::ImageView > outputImage;
    /// The index image for transform. Its dimensions are forced to the same values as the input. Its data type is forced to unsigned integer.
    std::shared_ptr< iolink::ImageView > outputTransformIndexImage;
    /// The data frame indicating the rotations and scales to apply.
    std::shared_ptr< const iolink::DataFrameView > transformTable;
};

/// Computes a similarity score for each voxel of a three-dimensional image for localizing a searched structure.
/// Warning: This command is experimental, his signature may be modified between now and his final version.
class IMAGEDEV_CPP_API CudaTemplateMatching3d final : public GenericAlgorithm
{
public:
    /// The way to manage the GPU memory. If the input image is already on GPU, this parameter is ignored.
    enum TilingMode
    {
    /// The entire input image is transferred to the GPU memory.
        NONE = 0,
    /// The input image is processed by tiles of a predefined size.
        USER_DEFINED
    };

    // Command constructor.
    CudaTemplateMatching3d();


    /// Gets the inputImage parameter.
    /// The input image where the template is searched.
    std::shared_ptr< iolink::ImageView > inputImage() const;
    /// Sets the inputImage parameter.
    /// The input image where the template is searched.
    void setInputImage( std::shared_ptr< iolink::ImageView > inputImage );

    /// Gets the inputTemplate parameter.
    /// The template image. Its shape must be contained in the input image.
    std::shared_ptr< iolink::ImageView > inputTemplate() const;
    /// Sets the inputTemplate parameter.
    /// The template image. Its shape must be contained in the input image.
    void setInputTemplate( std::shared_ptr< iolink::ImageView > inputTemplate );

    /// Gets the inputMask parameter.
    /// The input mask. Its shape must be the same as the template image. This parameter is optional.
    std::shared_ptr< iolink::ImageView > inputMask() const;
    /// Sets the inputMask parameter.
    /// The input mask. Its shape must be the same as the template image. This parameter is optional.
    void setInputMask( std::shared_ptr< iolink::ImageView > inputMask );

    /// Gets the inputFilter parameter.
    /// The input filter image used to correct the input template in the Fourier domain. Its spacing must be the same as the input image. This parameter is optional.
    std::shared_ptr< iolink::ImageView > inputFilter() const;
    /// Sets the inputFilter parameter.
    /// The input filter image used to correct the input template in the Fourier domain. Its spacing must be the same as the input image. This parameter is optional.
    void setInputFilter( std::shared_ptr< iolink::ImageView > inputFilter );

    /// Gets the rotateMask parameter.
    /// The flag that indicates if the mask has to be rotated with the template.
    bool rotateMask() const;
    /// Sets the rotateMask parameter.
    /// The flag that indicates if the mask has to be rotated with the template.
    void setRotateMask( const bool& rotateMask );

    /// Gets the transformTable parameter.
    /// The data frame indicating the rotations and scales to apply.
    std::shared_ptr< const iolink::DataFrameView > transformTable() const;
    /// Sets the transformTable parameter.
    /// The data frame indicating the rotations and scales to apply.
    void setTransformTable( const std::shared_ptr< const iolink::DataFrameView >& transformTable );

    /// Gets the tilingMode parameter.
    /// The way to manage the GPU memory. If the input image is already on GPU, this parameter is ignored.
    CudaTemplateMatching3d::TilingMode tilingMode() const;
    /// Sets the tilingMode parameter.
    /// The way to manage the GPU memory. If the input image is already on GPU, this parameter is ignored.
    void setTilingMode( const CudaTemplateMatching3d::TilingMode& tilingMode );

    /// Gets the tileSize parameter.
    /// The tile width and height in voxels. They must be greater than or equal to the correlation kernel width and height. This parameter is used only in USER_DEFINED tiling mode.
    iolink::Vector3u32 tileSize() const;
    /// Sets the tileSize parameter.
    /// The tile width and height in voxels. They must be greater than or equal to the correlation kernel width and height. This parameter is used only in USER_DEFINED tiling mode.
    void setTileSize( const iolink::Vector3u32& tileSize );

    /// Gets the cudaContext parameter.
    /// The CUDA context information.
    CudaContext::Ptr cudaContext() const;
    /// Sets the cudaContext parameter.
    /// The CUDA context information.
    void setCudaContext( const CudaContext::Ptr& cudaContext );

    /// Gets the outputImage parameter.
    /// The correlation scores image. Its dimensions are forced to the same values as the input. Its data type is forced to floating point.
    std::shared_ptr< iolink::ImageView > outputImage() const;
    /// Sets the outputImage parameter.
    /// The correlation scores image. Its dimensions are forced to the same values as the input. Its data type is forced to floating point.
    void setOutputImage( std::shared_ptr< iolink::ImageView > outputImage );

    /// Gets the outputTransformIndexImage parameter.
    /// The index image for transform. Its dimensions are forced to the same values as the input. Its data type is forced to unsigned integer.
    std::shared_ptr< iolink::ImageView > outputTransformIndexImage() const;
    /// Sets the outputTransformIndexImage parameter.
    /// The index image for transform. Its dimensions are forced to the same values as the input. Its data type is forced to unsigned integer.
    void setOutputTransformIndexImage( std::shared_ptr< iolink::ImageView > outputTransformIndexImage );

    // Method to launch the command.
    void execute();

};

/// Computes a similarity score for each voxel of a three-dimensional image for localizing a searched structure.
/// Warning: This command is experimental, his signature may be modified between now and his final version.
/// @param inputImage The input image where the template is searched.
/// @param inputTemplate The template image. Its shape must be contained in the input image.
/// @param inputMask The input mask. Its shape must be the same as the template image. This parameter is optional.
/// @param inputFilter The input filter image used to correct the input template in the Fourier domain. Its spacing must be the same as the input image. This parameter is optional.
/// @param rotateMask The flag that indicates if the mask has to be rotated with the template.
/// @param transformTable The data frame indicating the rotations and scales to apply.
/// @param tilingMode The way to manage the GPU memory. If the input image is already on GPU, this parameter is ignored.
/// @param tileSize The tile width and height in voxels. They must be greater than or equal to the correlation kernel width and height. This parameter is used only in USER_DEFINED tiling mode.
/// @param cudaContext The CUDA context information.
/// @param outputImage The correlation scores image. Its dimensions are forced to the same values as the input. Its data type is forced to floating point.
/// @param outputTransformIndexImage The index image for transform. Its dimensions are forced to the same values as the input. Its data type is forced to unsigned integer.
/// @return Returns a CudaTemplateMatching3dOutput structure containing the outputImage and outputTransformIndexImage output parameters.
IMAGEDEV_CPP_API 
CudaTemplateMatching3dOutput
cudaTemplateMatching3d( std::shared_ptr< iolink::ImageView > inputImage,
                        std::shared_ptr< iolink::ImageView > inputTemplate,
                        std::shared_ptr< iolink::ImageView > inputMask,
                        std::shared_ptr< iolink::ImageView > inputFilter,
                        bool rotateMask,
                        std::shared_ptr< const iolink::DataFrameView > transformTable,
                        CudaTemplateMatching3d::TilingMode tilingMode,
                        const iolink::Vector3u32& tileSize,
                        CudaContext::Ptr cudaContext,
                        std::shared_ptr< iolink::ImageView > outputImage = nullptr,
                        std::shared_ptr< iolink::ImageView > outputTransformIndexImage = nullptr );
} // namespace imagedev
