/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <iolink/ArrayX.h>
#include <memory>
#include <iolink/view/ImageView.h>

namespace imagedev
{
// Output structure of the dentalMeshExtraction3d function.
struct DentalMeshExtraction3dOutput
{
    /// The polyline representing the dental arch in the XY plan. This polyline must be represented as a two-dimensional array storing the coordinates of each vertex. If not set, it is automatically computed.
    std::shared_ptr<iolink::ArrayXd> inputArchCurve;
    /// The polyline representing the dental arch which is used to compute the arch mesh. This polyline is represented as a two-dimensional array storing the coordinates of each vertex. . If the input arch curve is not set, it is automatically computed from the input CBCT image. Otherwise, it corresponds to the resampled input arch curve. Curve vertex coordinates are expressed in mm.
    std::shared_ptr<iolink::ArrayXd> outputArchCurve;
    /// The polyline representing the inside border of the search area for fitting the arch mesh. This curve is computed by shifting the arch curve inside the jaws to the search distance. This polyline is represented as a two-dimensional array storing the coordinates of each vertex. Curve vertex coordinates are expressed in mm.
    std::shared_ptr<iolink::ArrayXd> outputInsideArchCurve;
    /// The polyline representing the outside border of the search area for fitting the arch mesh. This curve is computed by shifting the arch curve outside the jaws to the search distance. This polyline is represented as a two-dimensional array storing the coordinates of each vertex. Curve vertices are expressed in mm.
    std::shared_ptr<iolink::ArrayXd> outputOutsideArchCurve;
    /// The arch mesh vertex normals. These directions are calculated on the initial mesh prior to fitting. It is recommended to use these normals to unfold the tooth dental and generate a panoramic.  The fitted mesh vertex normals tend to cross each other and therefore generate artifacts.
    std::shared_ptr<iolink::ArrayXd> outputDirections;
    /// The quadrilateral mesh forming the surface fitted to the dental wall. This surface can be used to unfold the dental wall and generate a panoramic. This mesh is represented as a three-dimensional array storing the coordinates of each vertex.
    std::shared_ptr<iolink::ArrayXd> outputMesh;
};

/// Computes a quadrilateral mesh fitting on the dental wall defined in a three-dimensional binary image.
class IMAGEDEV_CPP_API DentalMeshExtraction3d final : public GenericAlgorithm
{
public:

    // Command constructor.
    DentalMeshExtraction3d();


    /// Gets the inputBinaryImage parameter.
    /// The input binary image representing the extracted dental wall. Its dimensions are equal to the input image dimensions divided by the sampling factor used at the segmentation step.
    std::shared_ptr< iolink::ImageView > inputBinaryImage() const;
    /// Sets the inputBinaryImage parameter.
    /// The input binary image representing the extracted dental wall. Its dimensions are equal to the input image dimensions divided by the sampling factor used at the segmentation step.
    void setInputBinaryImage( std::shared_ptr< iolink::ImageView > inputBinaryImage );

    /// Gets the inputImage parameter.
    /// The input 3D CBCT image of the head (LP orientation). It is used to compute automatically the arch curve. It is ignored if an input arch curve is user-defined
    std::shared_ptr< iolink::ImageView > inputImage() const;
    /// Sets the inputImage parameter.
    /// The input 3D CBCT image of the head (LP orientation). It is used to compute automatically the arch curve. It is ignored if an input arch curve is user-defined
    void setInputImage( std::shared_ptr< iolink::ImageView > inputImage );

    /// Gets the inputArchCurve parameter.
    /// The polyline representing the dental arch in the XY plan. This polyline must be represented as a two-dimensional array storing the coordinates of each vertex. If not set, it is automatically computed.
    std::shared_ptr<iolink::ArrayXd> inputArchCurve() const;
    /// Sets the inputArchCurve parameter.
    /// The polyline representing the dental arch in the XY plan. This polyline must be represented as a two-dimensional array storing the coordinates of each vertex. If not set, it is automatically computed.
    void setInputArchCurve( std::shared_ptr<iolink::ArrayXd> inputArchCurve );

    /// Gets the extrusionDown parameter.
    /// The extrusion length of dental arch curve in the direction of head towards feet. This length is expressed in mm.
    double extrusionDown() const;
    /// Sets the extrusionDown parameter.
    /// The extrusion length of dental arch curve in the direction of head towards feet. This length is expressed in mm.
    void setExtrusionDown( const double& extrusionDown );

    /// Gets the extrusionUp parameter.
    /// The extrusion length of dental arch curve in the direction of feet towards head. This length is expressed in mm.
    double extrusionUp() const;
    /// Sets the extrusionUp parameter.
    /// The extrusion length of dental arch curve in the direction of feet towards head. This length is expressed in mm.
    void setExtrusionUp( const double& extrusionUp );

    /// Gets the searchLength parameter.
    /// The search distance where the initial mesh is fitted to the arch curve. This distance is expressed in mm.
    double searchLength() const;
    /// Sets the searchLength parameter.
    /// The search distance where the initial mesh is fitted to the arch curve. This distance is expressed in mm.
    void setSearchLength( const double& searchLength );

    /// Gets the outputArchCurve parameter.
    /// The polyline representing the dental arch which is used to compute the arch mesh. This polyline is represented as a two-dimensional array storing the coordinates of each vertex. . If the input arch curve is not set, it is automatically computed from the input CBCT image. Otherwise, it corresponds to the resampled input arch curve. Curve vertex coordinates are expressed in mm.
    std::shared_ptr<iolink::ArrayXd> outputArchCurve() const;
    /// Sets the outputArchCurve parameter.
    /// The polyline representing the dental arch which is used to compute the arch mesh. This polyline is represented as a two-dimensional array storing the coordinates of each vertex. . If the input arch curve is not set, it is automatically computed from the input CBCT image. Otherwise, it corresponds to the resampled input arch curve. Curve vertex coordinates are expressed in mm.
    void setOutputArchCurve( std::shared_ptr<iolink::ArrayXd> outputArchCurve );

    /// Gets the outputInsideArchCurve parameter.
    /// The polyline representing the inside border of the search area for fitting the arch mesh. This curve is computed by shifting the arch curve inside the jaws to the search distance. This polyline is represented as a two-dimensional array storing the coordinates of each vertex. Curve vertex coordinates are expressed in mm.
    std::shared_ptr<iolink::ArrayXd> outputInsideArchCurve() const;
    /// Sets the outputInsideArchCurve parameter.
    /// The polyline representing the inside border of the search area for fitting the arch mesh. This curve is computed by shifting the arch curve inside the jaws to the search distance. This polyline is represented as a two-dimensional array storing the coordinates of each vertex. Curve vertex coordinates are expressed in mm.
    void setOutputInsideArchCurve( std::shared_ptr<iolink::ArrayXd> outputInsideArchCurve );

    /// Gets the outputOutsideArchCurve parameter.
    /// The polyline representing the outside border of the search area for fitting the arch mesh. This curve is computed by shifting the arch curve outside the jaws to the search distance. This polyline is represented as a two-dimensional array storing the coordinates of each vertex. Curve vertices are expressed in mm.
    std::shared_ptr<iolink::ArrayXd> outputOutsideArchCurve() const;
    /// Sets the outputOutsideArchCurve parameter.
    /// The polyline representing the outside border of the search area for fitting the arch mesh. This curve is computed by shifting the arch curve outside the jaws to the search distance. This polyline is represented as a two-dimensional array storing the coordinates of each vertex. Curve vertices are expressed in mm.
    void setOutputOutsideArchCurve( std::shared_ptr<iolink::ArrayXd> outputOutsideArchCurve );

    /// Gets the outputDirections parameter.
    /// The arch mesh vertex normals. These directions are calculated on the initial mesh prior to fitting. It is recommended to use these normals to unfold the tooth dental and generate a panoramic.  The fitted mesh vertex normals tend to cross each other and therefore generate artifacts.
    std::shared_ptr<iolink::ArrayXd> outputDirections() const;
    /// Sets the outputDirections parameter.
    /// The arch mesh vertex normals. These directions are calculated on the initial mesh prior to fitting. It is recommended to use these normals to unfold the tooth dental and generate a panoramic.  The fitted mesh vertex normals tend to cross each other and therefore generate artifacts.
    void setOutputDirections( std::shared_ptr<iolink::ArrayXd> outputDirections );

    /// Gets the outputMesh parameter.
    /// The quadrilateral mesh forming the surface fitted to the dental wall. This surface can be used to unfold the dental wall and generate a panoramic. This mesh is represented as a three-dimensional array storing the coordinates of each vertex.
    std::shared_ptr<iolink::ArrayXd> outputMesh() const;
    /// Sets the outputMesh parameter.
    /// The quadrilateral mesh forming the surface fitted to the dental wall. This surface can be used to unfold the dental wall and generate a panoramic. This mesh is represented as a three-dimensional array storing the coordinates of each vertex.
    void setOutputMesh( std::shared_ptr<iolink::ArrayXd> outputMesh );

    // Method to launch the command.
    void execute();

};

/// Computes a quadrilateral mesh fitting on the dental wall defined in a three-dimensional binary image.
/// @param inputBinaryImage The input binary image representing the extracted dental wall. Its dimensions are equal to the input image dimensions divided by the sampling factor used at the segmentation step.
/// @param inputImage The input 3D CBCT image of the head (LP orientation). It is used to compute automatically the arch curve. It is ignored if an input arch curve is user-defined
/// @param inputArchCurve The polyline representing the dental arch in the XY plan. This polyline must be represented as a two-dimensional array storing the coordinates of each vertex. If not set, it is automatically computed.
/// @param extrusionDown The extrusion length of dental arch curve in the direction of head towards feet. This length is expressed in mm.
/// @param extrusionUp The extrusion length of dental arch curve in the direction of feet towards head. This length is expressed in mm.
/// @param searchLength The search distance where the initial mesh is fitted to the arch curve. This distance is expressed in mm.
/// @param outputArchCurve The polyline representing the dental arch which is used to compute the arch mesh. This polyline is represented as a two-dimensional array storing the coordinates of each vertex. . If the input arch curve is not set, it is automatically computed from the input CBCT image. Otherwise, it corresponds to the resampled input arch curve. Curve vertex coordinates are expressed in mm.
/// @param outputInsideArchCurve The polyline representing the inside border of the search area for fitting the arch mesh. This curve is computed by shifting the arch curve inside the jaws to the search distance. This polyline is represented as a two-dimensional array storing the coordinates of each vertex. Curve vertex coordinates are expressed in mm.
/// @param outputOutsideArchCurve The polyline representing the outside border of the search area for fitting the arch mesh. This curve is computed by shifting the arch curve outside the jaws to the search distance. This polyline is represented as a two-dimensional array storing the coordinates of each vertex. Curve vertices are expressed in mm.
/// @param outputDirections The arch mesh vertex normals. These directions are calculated on the initial mesh prior to fitting. It is recommended to use these normals to unfold the tooth dental and generate a panoramic.  The fitted mesh vertex normals tend to cross each other and therefore generate artifacts.
/// @param outputMesh The quadrilateral mesh forming the surface fitted to the dental wall. This surface can be used to unfold the dental wall and generate a panoramic. This mesh is represented as a three-dimensional array storing the coordinates of each vertex.
/// @return Returns a DentalMeshExtraction3dOutput structure containing the outputArchCurve, outputInsideArchCurve, outputOutsideArchCurve, outputDirections and outputMesh output parameters.
IMAGEDEV_CPP_API 
DentalMeshExtraction3dOutput
dentalMeshExtraction3d( std::shared_ptr< iolink::ImageView > inputBinaryImage,
                        std::shared_ptr< iolink::ImageView > inputImage,
                        std::shared_ptr<iolink::ArrayXd> inputArchCurve,
                        double extrusionDown,
                        double extrusionUp,
                        double searchLength,
                        std::shared_ptr<iolink::ArrayXd> outputArchCurve = nullptr,
                        std::shared_ptr<iolink::ArrayXd> outputInsideArchCurve = nullptr,
                        std::shared_ptr<iolink::ArrayXd> outputOutsideArchCurve = nullptr,
                        std::shared_ptr<iolink::ArrayXd> outputDirections = nullptr,
                        std::shared_ptr<iolink::ArrayXd> outputMesh = nullptr );
} // namespace imagedev
