/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <iolink/view/ImageView.h>

namespace imagedev
{
/// Transforms a grayscale CBCT image into a binary image where every pixel identified as belonging to the dental wall is set to 1 and all the others are set to 0.
class IMAGEDEV_CPP_API DentalWallSegmentation3d final : public GenericAlgorithm
{
public:

    // Command constructor.
    DentalWallSegmentation3d();


    /// Gets the inputImage parameter.
    /// The input 3D CBCT image of the head (LP orientation).
    std::shared_ptr< iolink::ImageView > inputImage() const;
    /// Sets the inputImage parameter.
    /// The input 3D CBCT image of the head (LP orientation).
    void setInputImage( std::shared_ptr< iolink::ImageView > inputImage );

    /// Gets the autoThresholdMode parameter.
    /// The way to binarize the data. Performs an automatic segmentation of the teeth is equals true, applies the threshold defined by the thresholdRange parameter otherwise.
    bool autoThresholdMode() const;
    /// Sets the autoThresholdMode parameter.
    /// The way to binarize the data. Performs an automatic segmentation of the teeth is equals true, applies the threshold defined by the thresholdRange parameter otherwise.
    void setAutoThresholdMode( const bool& autoThresholdMode );

    /// Gets the samplingFactor parameter.
    /// The down sampling factor to compute the segmentation volume used for extracting the dental wall.
    uint32_t samplingFactor() const;
    /// Sets the samplingFactor parameter.
    /// The down sampling factor to compute the segmentation volume used for extracting the dental wall.
    void setSamplingFactor( const uint32_t& samplingFactor );

    /// Gets the threshold parameter.
    /// The low and high threshold levels. This parameter is ignored when the automatic threshold mode is enabled.
    double threshold() const;
    /// Sets the threshold parameter.
    /// The low and high threshold levels. This parameter is ignored when the automatic threshold mode is enabled.
    void setThreshold( const double& threshold );

    /// Gets the outputBinaryImage parameter.
    /// The output binary image representing the extracted dental wall. Its dimensions are equal to the input image dimensions divided by the sampling factor.
    std::shared_ptr< iolink::ImageView > outputBinaryImage() const;
    /// Sets the outputBinaryImage parameter.
    /// The output binary image representing the extracted dental wall. Its dimensions are equal to the input image dimensions divided by the sampling factor.
    void setOutputBinaryImage( std::shared_ptr< iolink::ImageView > outputBinaryImage );

    // Method to launch the command.
    void execute();

};

/// Transforms a grayscale CBCT image into a binary image where every pixel identified as belonging to the dental wall is set to 1 and all the others are set to 0.
/// @param inputImage The input 3D CBCT image of the head (LP orientation).
/// @param autoThresholdMode The way to binarize the data. Performs an automatic segmentation of the teeth is equals true, applies the threshold defined by the thresholdRange parameter otherwise.
/// @param samplingFactor The down sampling factor to compute the segmentation volume used for extracting the dental wall.
/// @param threshold The low and high threshold levels. This parameter is ignored when the automatic threshold mode is enabled.
/// @param outputBinaryImage The output binary image representing the extracted dental wall. Its dimensions are equal to the input image dimensions divided by the sampling factor.
/// @return Returns the outputBinaryImage output parameter.
IMAGEDEV_CPP_API 
std::shared_ptr< iolink::ImageView >
dentalWallSegmentation3d( std::shared_ptr< iolink::ImageView > inputImage,
                          bool autoThresholdMode,
                          uint32_t samplingFactor,
                          double threshold,
                          std::shared_ptr< iolink::ImageView > outputBinaryImage = nullptr );
} // namespace imagedev
