/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <iolink/view/ImageView.h>

namespace imagedev
{
/// Computes the Euclidean distance map of objects in a three-dimensional binary image.
class IMAGEDEV_CPP_API EuclideanDistanceMap3d final : public GenericAlgorithm
{
public:
    /// The mode defining where the distance field is computed.
    enum MappingMode
    {
    /// The distances are computed for voxels inside the objects (having an intensity equal to 1). Outside voxels remain at zero.
        INSIDE = 0,
    /// The distances are computed for voxels outside of objects (having an intensity equal to 0). Inside voxels are set to zero.
        OUTSIDE
    };
    /// The mode defining the border conditions.
    enum BorderCondition
    {
    /// Voxels outside of the image are considered to have the value 0.
        ZERO = 0,
    /// Voxels outside of the image are considered as an extrapolation of the image voxels with a mirror border condition.
        MIRROR
    };

    // Command constructor.
    EuclideanDistanceMap3d();


    /// Gets the inputBinaryImage parameter.
    /// The binary input image.
    std::shared_ptr< iolink::ImageView > inputBinaryImage() const;
    /// Sets the inputBinaryImage parameter.
    /// The binary input image.
    void setInputBinaryImage( std::shared_ptr< iolink::ImageView > inputBinaryImage );

    /// Gets the mappingMode parameter.
    /// The mode defining where the distance field is computed.
    EuclideanDistanceMap3d::MappingMode mappingMode() const;
    /// Sets the mappingMode parameter.
    /// The mode defining where the distance field is computed.
    void setMappingMode( const EuclideanDistanceMap3d::MappingMode& mappingMode );

    /// Gets the borderCondition parameter.
    /// The mode defining the border conditions.
    EuclideanDistanceMap3d::BorderCondition borderCondition() const;
    /// Sets the borderCondition parameter.
    /// The mode defining the border conditions.
    void setBorderCondition( const EuclideanDistanceMap3d::BorderCondition& borderCondition );

    /// Gets the outputMapImage parameter.
    /// The distance map output image. Distance map of the shape represented by the input image. Each voxel is assigned to a value depending on the distance to the nearest object boundary. Its dimensions are forced to the same values as the input. Its data type is 32-bit floating point.
    std::shared_ptr< iolink::ImageView > outputMapImage() const;
    /// Sets the outputMapImage parameter.
    /// The distance map output image. Distance map of the shape represented by the input image. Each voxel is assigned to a value depending on the distance to the nearest object boundary. Its dimensions are forced to the same values as the input. Its data type is 32-bit floating point.
    void setOutputMapImage( std::shared_ptr< iolink::ImageView > outputMapImage );

    // Method to launch the command.
    void execute();

};

/// Computes the Euclidean distance map of objects in a three-dimensional binary image.
/// @param inputBinaryImage The binary input image.
/// @param mappingMode The mode defining where the distance field is computed.
/// @param borderCondition The mode defining the border conditions.
/// @param outputMapImage The distance map output image. Distance map of the shape represented by the input image. Each voxel is assigned to a value depending on the distance to the nearest object boundary. Its dimensions are forced to the same values as the input. Its data type is 32-bit floating point.
/// @return Returns the outputMapImage output parameter.
IMAGEDEV_CPP_API 
std::shared_ptr< iolink::ImageView >
euclideanDistanceMap3d( std::shared_ptr< iolink::ImageView > inputBinaryImage,
                        EuclideanDistanceMap3d::MappingMode mappingMode,
                        EuclideanDistanceMap3d::BorderCondition borderCondition,
                        std::shared_ptr< iolink::ImageView > outputMapImage = nullptr );
} // namespace imagedev
