/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <iolink/view/ImageView.h>
#include <iolink/Matrix.h>

namespace imagedev
{
/// Estimates a transformation (translation and rotation) that roughly aligns a moving image with a fixed image based on their intensity values.
class IMAGEDEV_CPP_API ImagePrealignment3d final : public GenericAlgorithm
{
public:
    /// The type of pre-alignment to perform.
    enum AlignmentMode
    {
    /// In this mode, the centers of gravity of the input images are aligned.
        CENTERS = 0,
    /// In this mode, the centers of gravity and the principal axes of the input images are aligned.
        PRINCIPAL_AXES
    };

    // Command constructor.
    ImagePrealignment3d();


    /// Gets the inputMovingImage parameter.
    /// The input moving image, also known as the model image.
    std::shared_ptr< iolink::ImageView > inputMovingImage() const;
    /// Sets the inputMovingImage parameter.
    /// The input moving image, also known as the model image.
    void setInputMovingImage( std::shared_ptr< iolink::ImageView > inputMovingImage );

    /// Gets the inputFixedImage parameter.
    /// The input fixed image, also known as the reference image.
    std::shared_ptr< iolink::ImageView > inputFixedImage() const;
    /// Sets the inputFixedImage parameter.
    /// The input fixed image, also known as the reference image.
    void setInputFixedImage( std::shared_ptr< iolink::ImageView > inputFixedImage );

    /// Gets the alignmentMode parameter.
    /// The type of pre-alignment to perform.
    ImagePrealignment3d::AlignmentMode alignmentMode() const;
    /// Sets the alignmentMode parameter.
    /// The type of pre-alignment to perform.
    void setAlignmentMode( const ImagePrealignment3d::AlignmentMode& alignmentMode );

    /// Gets the outputTransform parameter.
    /// The output transformation, which pre-aligns the moving image onto the fixed image. It is represented by a 4x4 matrix.
    iolink::Matrix4d outputTransform() const;

    // Method to launch the command.
    void execute();

};

/// Estimates a transformation (translation and rotation) that roughly aligns a moving image with a fixed image based on their intensity values.
/// @param inputMovingImage The input moving image, also known as the model image.
/// @param inputFixedImage The input fixed image, also known as the reference image.
/// @param alignmentMode The type of pre-alignment to perform.
/// @param outputTransform The output transformation, which pre-aligns the moving image onto the fixed image. It is represented by a 4x4 matrix.
/// @return Returns the outputTransform output parameter.
IMAGEDEV_CPP_API 
iolink::Matrix4d
imagePrealignment3d( std::shared_ptr< iolink::ImageView > inputMovingImage,
                     std::shared_ptr< iolink::ImageView > inputFixedImage,
                     ImagePrealignment3d::AlignmentMode alignmentMode );
} // namespace imagedev
