/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <iolink/view/ImageView.h>

namespace imagedev
{
// Output structure of the imageStackProjection3d function.
struct ImageStackProjection3dOutput
{
    /// The output image representing the volume projection according to the selected criterion.
    std::shared_ptr< iolink::ImageView > outputImage;
    /// The output map label image. Each pixel intensity represents the Z index used in the output image.
    std::shared_ptr< iolink::ImageView > outputMapImage;
};

/// Transforms a volume that represents a stack of 2D images into a single 2D image.
class IMAGEDEV_CPP_API ImageStackProjection3d final : public GenericAlgorithm
{
public:
    /// The projection criterion (the rule used to transform a stack of pixels into a single pixel).
    enum ProjectionMode
    {
    /// The pixel of highest intensity is selected in the stack. The map output is available with this mode.
        INTENSITY_MAXIMA = 0,
    /// The pixel of highest gradient is selected in the stack. The gradient is computed as defined by the gradientOperator parameter. The map output is available with this mode.
        GRADIENT_MAXIMA,
    /// The pixels of the stack are added up. If the autoScale mode is enabled, the result is normalized by the number of images of the stack, which is exactly the same as using the MEAN criterion. The map output is not available with this mode.
        PROJECTION,
    /// The pixel of lowest intensity is selected in the stack. The map output is available with this mode.
        INTENSITY_MINIMA,
    /// The pixel of median intensity is selected in the stack. The map output is not available with this mode.
        MEDIAN,
    /// The pixels of the stack are averaged. The map output is not available with this mode.
        MEAN,
    /// The output value is the standard deviation of the stack values. The map output is not available with this mode.
        VARIANCE,
    /// The output value is the entropy of the stack values, based on this formula: $$ Entropy= -\sum_{i \in seq} {P(x_i)log(P(x_i))} $$ where $P(x_i)$ is the probability to have a pixel with value $x_i$ (it's the number of those pixels divided by the total number of pixels). The map output is not available with this mode.
        ENTROPY,
    /// The output value is the energy of the stack values, based on this formula: $$ Energy= \sum_{i \in seq} {P(x_i)^2} $$ The map output is not available with this mode.
        ENERGY
    };
    /// The automatic intensity scaling mode to average the pixels of the stack. It is used only with the PROJECTION mode and is ignored with any other criterion.
    enum AutoScale
    {
    /// The sum of intensities is not divided by the number of images in the stack.
        NO = 0,
    /// The sum of intensities is divided by the number of images in the stack.
        YES
    };
    /// The gradient operator to apply.It is used only with the GRADIENT_MAXIMA mode and is ignored with any other criterion.
    enum GradientOperator
    {
    /// A gaussian gradient is performed (convolution with derivatives of gaussian function along the stack direction).
        GAUSSIAN_GRADIENT = 0,
    /// A morphological gradient is performed (see MorphologicalGradient for definition). This is the recommended operator to use.
        MORPHOLOGICAL_GRADIENT,
    /// A recursive gradient is performed (Canny-Deriche on stack direction).
        RECURSIVE_GRADIENT
    };

    // Command constructor.
    ImageStackProjection3d();


    /// Gets the inputImage parameter.
    /// A 3D volume representing a stack of 2D images.
    std::shared_ptr< iolink::ImageView > inputImage() const;
    /// Sets the inputImage parameter.
    /// A 3D volume representing a stack of 2D images.
    void setInputImage( std::shared_ptr< iolink::ImageView > inputImage );

    /// Gets the projectionMode parameter.
    /// The projection criterion (the rule used to transform a stack of pixels into a single pixel).
    ImageStackProjection3d::ProjectionMode projectionMode() const;
    /// Sets the projectionMode parameter.
    /// The projection criterion (the rule used to transform a stack of pixels into a single pixel).
    void setProjectionMode( const ImageStackProjection3d::ProjectionMode& projectionMode );

    /// Gets the autoScale parameter.
    /// The automatic intensity scaling mode to average the pixels of the stack. It is used only with the PROJECTION mode and is ignored with any other criterion.
    ImageStackProjection3d::AutoScale autoScale() const;
    /// Sets the autoScale parameter.
    /// The automatic intensity scaling mode to average the pixels of the stack. It is used only with the PROJECTION mode and is ignored with any other criterion.
    void setAutoScale( const ImageStackProjection3d::AutoScale& autoScale );

    /// Gets the smoothingSize parameter.
    /// The kernel size for smoothing the gradient before selecting the output value. It is used only with the GRADIENT_MAXIMA mode and is ignored with any other criterion.
    int32_t smoothingSize() const;
    /// Sets the smoothingSize parameter.
    /// The kernel size for smoothing the gradient before selecting the output value. It is used only with the GRADIENT_MAXIMA mode and is ignored with any other criterion.
    void setSmoothingSize( const int32_t& smoothingSize );

    /// Gets the gradientOperator parameter.
    /// The gradient operator to apply.It is used only with the GRADIENT_MAXIMA mode and is ignored with any other criterion.
    ImageStackProjection3d::GradientOperator gradientOperator() const;
    /// Sets the gradientOperator parameter.
    /// The gradient operator to apply.It is used only with the GRADIENT_MAXIMA mode and is ignored with any other criterion.
    void setGradientOperator( const ImageStackProjection3d::GradientOperator& gradientOperator );

    /// Gets the outputImage parameter.
    /// The output image representing the volume projection according to the selected criterion.
    std::shared_ptr< iolink::ImageView > outputImage() const;
    /// Sets the outputImage parameter.
    /// The output image representing the volume projection according to the selected criterion.
    void setOutputImage( std::shared_ptr< iolink::ImageView > outputImage );

    /// Gets the outputMapImage parameter.
    /// The output map label image. Each pixel intensity represents the Z index used in the output image.
    std::shared_ptr< iolink::ImageView > outputMapImage() const;
    /// Sets the outputMapImage parameter.
    /// The output map label image. Each pixel intensity represents the Z index used in the output image.
    void setOutputMapImage( std::shared_ptr< iolink::ImageView > outputMapImage );

    // Method to launch the command.
    void execute();

};

/// Transforms a volume that represents a stack of 2D images into a single 2D image.
/// @param inputImage A 3D volume representing a stack of 2D images.
/// @param projectionMode The projection criterion (the rule used to transform a stack of pixels into a single pixel).
/// @param autoScale The automatic intensity scaling mode to average the pixels of the stack. It is used only with the PROJECTION mode and is ignored with any other criterion.
/// @param smoothingSize The kernel size for smoothing the gradient before selecting the output value. It is used only with the GRADIENT_MAXIMA mode and is ignored with any other criterion.
/// @param gradientOperator The gradient operator to apply.It is used only with the GRADIENT_MAXIMA mode and is ignored with any other criterion.
/// @param outputImage The output image representing the volume projection according to the selected criterion.
/// @param outputMapImage The output map label image. Each pixel intensity represents the Z index used in the output image.
/// @return Returns a ImageStackProjection3dOutput structure containing the outputImage and outputMapImage output parameters.
IMAGEDEV_CPP_API 
ImageStackProjection3dOutput
imageStackProjection3d( std::shared_ptr< iolink::ImageView > inputImage,
                        ImageStackProjection3d::ProjectionMode projectionMode,
                        ImageStackProjection3d::AutoScale autoScale,
                        int32_t smoothingSize,
                        ImageStackProjection3d::GradientOperator gradientOperator,
                        std::shared_ptr< iolink::ImageView > outputImage = nullptr,
                        std::shared_ptr< iolink::ImageView > outputMapImage = nullptr );
} // namespace imagedev
