/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <iolink/view/ImageView.h>

namespace imagedev
{
/// Performs a pointwise logical operation between an image and a value.
class IMAGEDEV_CPP_API LogicalOperationWithValue final : public GenericAlgorithm
{
public:
    /// The logical operator to apply. Default is AND.
    enum LogicalOperator
    {
    /// The logical AND operator performs a bitwise conjunction between an image and a constant value.
        AND = 0,
    /// The logical OR operator performs a bitwise disjunction between an image and a constant value.
        OR,
    /// The logical XOR operator performs a bitwise exclusive disjunction between an image and a constant value.
        XOR,
    /// The logical NAND operator performs a bitwise alternative denial between an image and a constant value.
        NAND,
    /// The logical NOR operator performs a bitwise joint denial between an image and a constant value.
        NOR,
    /// The logical NXOR operator performs a bitwise equivalence (or logical biconditional) between an image and a constant value.
        NXOR,
    /// The logical SUBTRACT operator performs a bitwise logical difference between an image and a constant value.
        SUBTRACT
    };

    // Command constructor.
    LogicalOperationWithValue();


    /// Gets the inputImage parameter.
    /// The input image, must be integer, including grayscale, binary, label, and color.
    std::shared_ptr< iolink::ImageView > inputImage() const;
    /// Sets the inputImage parameter.
    /// The input image, must be integer, including grayscale, binary, label, and color.
    void setInputImage( std::shared_ptr< iolink::ImageView > inputImage );

    /// Gets the value parameter.
    /// The value to apply, cast to the type of the input image.
    uint32_t value() const;
    /// Sets the value parameter.
    /// The value to apply, cast to the type of the input image.
    void setValue( const uint32_t& value );

    /// Gets the logicalOperator parameter.
    /// The logical operator to apply. Default is AND.
    LogicalOperationWithValue::LogicalOperator logicalOperator() const;
    /// Sets the logicalOperator parameter.
    /// The logical operator to apply. Default is AND.
    void setLogicalOperator( const LogicalOperationWithValue::LogicalOperator& logicalOperator );

    /// Gets the outputImage parameter.
    /// The output image, size and type are forced to the same values as the input.
    std::shared_ptr< iolink::ImageView > outputImage() const;
    /// Sets the outputImage parameter.
    /// The output image, size and type are forced to the same values as the input.
    void setOutputImage( std::shared_ptr< iolink::ImageView > outputImage );

    // Method to launch the command.
    void execute();

};

/// Performs a pointwise logical operation between an image and a value.
/// @param inputImage The input image, must be integer, including grayscale, binary, label, and color.
/// @param value The value to apply, cast to the type of the input image.
/// @param logicalOperator The logical operator to apply. Default is AND.
/// @param outputImage The output image, size and type are forced to the same values as the input.
/// @return Returns the outputImage output parameter.
IMAGEDEV_CPP_API 
std::shared_ptr< iolink::ImageView >
logicalOperationWithValue( std::shared_ptr< iolink::ImageView > inputImage,
                           uint32_t value,
                           LogicalOperationWithValue::LogicalOperator logicalOperator,
                           std::shared_ptr< iolink::ImageView > outputImage = nullptr );
} // namespace imagedev
