/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <iolink/view/ImageView.h>

namespace imagedev
{
/// Separates connected objects in a binary image.
class IMAGEDEV_CPP_API SeparateObjects final : public GenericAlgorithm
{
public:
    /// The type of result image.
    enum OutputType
    {
    /// The result image is a binary image representing the original data minus separation lines (split particles).
        SEPARATED_OBJECTS = 0,
    /// The result image is a binary image representing the separation lines (watershed).
        WATERSHED_RIDGES,
    /// The result image is a label image representing the catchment areas (labeled complement of the watershed).
        SEPARATED_BASINS,
    /// The result image is a label image representing the catchment areas with separation lines filled by a label.
        CONTIGUOUS_BASINS,
    /// The result image is a label image representing the original image with labels taking into account the watershed.
        CONTIGUOUS_OBJECTS
    };
    /// The mode for applying the watershed algorithm.
    enum AlgorithmMode
    {
    /// The result is repeatable but slower to compute.
        REPEATABLE = 0,
    /// The result is faster to compute but not repeatable because of asynchronous parallel computation. Since a watershed problem does not generally have a unique solution, two processings of the same image can lead to two different results (both exact).
        FAST
    };
    /// The 3D neighborhood configuration. This parameter is ignored with a 2D input image.
    enum Neighborhood
    {
    /// The structuring element is composed of voxels with a common face with the voxel of interest.
        CONNECTIVITY_6 = 0,
    /// The structuring element is composed of voxels with at least one common edge.
        CONNECTIVITY_18,
    /// The structuring element is a full cube.
        CONNECTIVITY_26
    };

    // Command constructor.
    SeparateObjects();


    /// Gets the inputBinaryImage parameter.
    /// The input binary image.
    std::shared_ptr< iolink::ImageView > inputBinaryImage() const;
    /// Sets the inputBinaryImage parameter.
    /// The input binary image.
    void setInputBinaryImage( std::shared_ptr< iolink::ImageView > inputBinaryImage );

    /// Gets the contrastValue parameter.
    /// The depth of the valley used to select the markers of the watershed.
    int32_t contrastValue() const;
    /// Sets the contrastValue parameter.
    /// The depth of the valley used to select the markers of the watershed.
    void setContrastValue( const int32_t& contrastValue );

    /// Gets the outputType parameter.
    /// The type of result image.
    SeparateObjects::OutputType outputType() const;
    /// Sets the outputType parameter.
    /// The type of result image.
    void setOutputType( const SeparateObjects::OutputType& outputType );

    /// Gets the algorithmMode parameter.
    /// The mode for applying the watershed algorithm.
    SeparateObjects::AlgorithmMode algorithmMode() const;
    /// Sets the algorithmMode parameter.
    /// The mode for applying the watershed algorithm.
    void setAlgorithmMode( const SeparateObjects::AlgorithmMode& algorithmMode );

    /// Gets the neighborhood parameter.
    /// The 3D neighborhood configuration. This parameter is ignored with a 2D input image.
    SeparateObjects::Neighborhood neighborhood() const;
    /// Sets the neighborhood parameter.
    /// The 3D neighborhood configuration. This parameter is ignored with a 2D input image.
    void setNeighborhood( const SeparateObjects::Neighborhood& neighborhood );

    /// Gets the outputImage parameter.
    /// The output binary or label image. Its dimensions are forced to the same values as the input image. Its type depends on the outputType parameter.
    std::shared_ptr< iolink::ImageView > outputImage() const;
    /// Sets the outputImage parameter.
    /// The output binary or label image. Its dimensions are forced to the same values as the input image. Its type depends on the outputType parameter.
    void setOutputImage( std::shared_ptr< iolink::ImageView > outputImage );

    // Method to launch the command.
    void execute();

};

/// Separates connected objects in a binary image.
/// @param inputBinaryImage The input binary image.
/// @param contrastValue The depth of the valley used to select the markers of the watershed.
/// @param outputType The type of result image.
/// @param algorithmMode The mode for applying the watershed algorithm.
/// @param neighborhood The 3D neighborhood configuration. This parameter is ignored with a 2D input image.
/// @param outputImage The output binary or label image. Its dimensions are forced to the same values as the input image. Its type depends on the outputType parameter.
/// @return Returns the outputImage output parameter.
IMAGEDEV_CPP_API 
std::shared_ptr< iolink::ImageView >
separateObjects( std::shared_ptr< iolink::ImageView > inputBinaryImage,
                 int32_t contrastValue,
                 SeparateObjects::OutputType outputType,
                 SeparateObjects::AlgorithmMode algorithmMode,
                 SeparateObjects::Neighborhood neighborhood,
                 std::shared_ptr< iolink::ImageView > outputImage = nullptr );
} // namespace imagedev
