/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <iolink/Vector.h>
#include <iolink/view/ImageView.h>

namespace imagedev
{
// Output structure of the supervisedTextureClassification3d function.
struct SupervisedTextureClassification3dOutput
{
    /// The output label image representing the texture classification result. Its dimensions and type are forced to the same values as the training input.
    std::shared_ptr< iolink::ImageView > outputLabelImage;
    /// The output map image. Its dimensions are forced to the same values as the training image. In CLASS_DISTANCE mode, its number of channels is equal to the number of classes defined in the training image. Its data type is forced to floating point.
    std::shared_ptr< iolink::ImageView > outputMapImage;
};

/// Performs a segmentation of a three-dimensional grayscale image, based on a texture model automatically built from a training input image.
class IMAGEDEV_CPP_API SupervisedTextureClassification3d final : public GenericAlgorithm
{
public:
    /// The groups of textural features to compute. This list defines all the textural attributes proposed for performing the classification.
    enum FeatureGroup
    {
    /// Features based on co-occurrence matrices. One feature is extracted from each co-occurrence vector.
        DIRECTIONAL_COOCCURRENCE = 1,
    /// Features based on co-occurrence matrices. Three statistical features are extracted from all vectors.
        ROTATION_INVARIANT_COOCCURRENCE = 2,
    /// Features based on first order statistics that are not computed using a histogram.
        FIRST_ORDER_STATISTICS = 4,
    /// Features based on histogram statistics, including histogram quantiles.
        HISTOGRAM_STATISTICS = 8,
    /// Feature based on the intensity value of the input image.
        INTENSITY = 16
    };
    /// The shape of the co-occurrence texton (the pattern defined by the set of co-occurrence vectors). This parameter is ignored if none of the co-occurrence feature groups is selected.
    enum CoocTextonShape
    {
    /// The set of all points associated to corners, edge centers, and face centers of a cube with a half side size defined by the coocTextonSize parameter.
        CUBE = 0,
    /// The set of all the points located at the same euclidean distance defined by the coocTextonSize parameter from the center. This mode is recommended when a repetitive texture is mono-scale.
        SPHERE,
    /// The set of all the points located at a distance less or equal to the coocTextonSize parameter from the center. This mode can be useful to classify a multi-scale repetitive texture, but may be very time consuming.
        BALL
    };
    /// The type of uncertainty map image to compute.
    enum OutputMapType
    {
    /// The uncertainty map represents the Mahalanobis distance to the class selected by the classification. The closer to 0 this metric is, the more confident the classification.
        CLOSEST_DISTANCE = 0,
    /// The uncertainty map represents the Mahalanobis distance to the class selected by the classification (d1) weighted by the gap with the second closest distance (d2). The smaller this metric is, the more confident and less ambiguous the classification.
        RELATIVE_DISTANCE,
    /// The uncertainty map is a multichannel image where each channel represents the distance to the corresponding class.
        CLASS_DISTANCE,
    /// No uncertainty map is computed.
        NONE
    };

    // Command constructor.
    SupervisedTextureClassification3d();


    /// Gets the inputImage parameter.
    /// The input grayscale image to segment.
    std::shared_ptr< iolink::ImageView > inputImage() const;
    /// Sets the inputImage parameter.
    /// The input grayscale image to segment.
    void setInputImage( std::shared_ptr< iolink::ImageView > inputImage );

    /// Gets the inputTrainingImage parameter.
    /// The input label training image (16 or 32 bits) where each label represents a class sample for the training step.
    std::shared_ptr< iolink::ImageView > inputTrainingImage() const;
    /// Sets the inputTrainingImage parameter.
    /// The input label training image (16 or 32 bits) where each label represents a class sample for the training step.
    void setInputTrainingImage( std::shared_ptr< iolink::ImageView > inputTrainingImage );

    /// Gets the featureGroup parameter.
    /// The groups of textural features to compute. This list defines all the textural attributes proposed for performing the classification.
    int32_t featureGroup() const;
    /// Sets the featureGroup parameter.
    /// The groups of textural features to compute. This list defines all the textural attributes proposed for performing the classification.
    void setFeatureGroup( const int32_t& featureGroup );

    /// Gets the radiusRange parameter.
    /// The minimum and maximum radius, in voxels, of the circular neighborhoods used for computing textural features.
    iolink::Vector2u32 radiusRange() const;
    /// Sets the radiusRange parameter.
    /// The minimum and maximum radius, in voxels, of the circular neighborhoods used for computing textural features.
    void setRadiusRange( const iolink::Vector2u32& radiusRange );

    /// Gets the radiusStep parameter.
    /// The step, in voxels, used to define the set of radius between minimum and maximum. The maximum radius is systematically added to the radius list.
    uint32_t radiusStep() const;
    /// Sets the radiusStep parameter.
    /// The step, in voxels, used to define the set of radius between minimum and maximum. The maximum radius is systematically added to the radius list.
    void setRadiusStep( const uint32_t& radiusStep );

    /// Gets the coocRadius parameter.
    /// The radius, in voxels, of the circular neighborhood used by the co-occurrence features. This parameter is ignored if none of the co-occurrence feature groups is selected.
    uint32_t coocRadius() const;
    /// Sets the coocRadius parameter.
    /// The radius, in voxels, of the circular neighborhood used by the co-occurrence features. This parameter is ignored if none of the co-occurrence feature groups is selected.
    void setCoocRadius( const uint32_t& coocRadius );

    /// Gets the coocTextonShape parameter.
    /// The shape of the co-occurrence texton (the pattern defined by the set of co-occurrence vectors). This parameter is ignored if none of the co-occurrence feature groups is selected.
    SupervisedTextureClassification3d::CoocTextonShape coocTextonShape() const;
    /// Sets the coocTextonShape parameter.
    /// The shape of the co-occurrence texton (the pattern defined by the set of co-occurrence vectors). This parameter is ignored if none of the co-occurrence feature groups is selected.
    void setCoocTextonShape( const SupervisedTextureClassification3d::CoocTextonShape& coocTextonShape );

    /// Gets the coocTextonSize parameter.
    /// The size, in voxels, of the texton shape for co-occurrence features. This parameter is ignored if none of the co-occurrence feature groups is selected.
    uint32_t coocTextonSize() const;
    /// Sets the coocTextonSize parameter.
    /// The size, in voxels, of the texton shape for co-occurrence features. This parameter is ignored if none of the co-occurrence feature groups is selected.
    void setCoocTextonSize( const uint32_t& coocTextonSize );

    /// Gets the minSeparationPercentage parameter.
    /// This parameter controls the rejection criteria of the feature selection algorithm (FS).
    double minSeparationPercentage() const;
    /// Sets the minSeparationPercentage parameter.
    /// This parameter controls the rejection criteria of the feature selection algorithm (FS).
    void setMinSeparationPercentage( const double& minSeparationPercentage );

    /// Gets the outputMapType parameter.
    /// The type of uncertainty map image to compute.
    SupervisedTextureClassification3d::OutputMapType outputMapType() const;
    /// Sets the outputMapType parameter.
    /// The type of uncertainty map image to compute.
    void setOutputMapType( const SupervisedTextureClassification3d::OutputMapType& outputMapType );

    /// Gets the outputLabelImage parameter.
    /// The output label image representing the texture classification result. Its dimensions and type are forced to the same values as the training input.
    std::shared_ptr< iolink::ImageView > outputLabelImage() const;
    /// Sets the outputLabelImage parameter.
    /// The output label image representing the texture classification result. Its dimensions and type are forced to the same values as the training input.
    void setOutputLabelImage( std::shared_ptr< iolink::ImageView > outputLabelImage );

    /// Gets the outputMapImage parameter.
    /// The output map image. Its dimensions are forced to the same values as the training image. In CLASS_DISTANCE mode, its number of channels is equal to the number of classes defined in the training image. Its data type is forced to floating point.
    std::shared_ptr< iolink::ImageView > outputMapImage() const;
    /// Sets the outputMapImage parameter.
    /// The output map image. Its dimensions are forced to the same values as the training image. In CLASS_DISTANCE mode, its number of channels is equal to the number of classes defined in the training image. Its data type is forced to floating point.
    void setOutputMapImage( std::shared_ptr< iolink::ImageView > outputMapImage );

    // Method to launch the command.
    void execute();

};

/// Performs a segmentation of a three-dimensional grayscale image, based on a texture model automatically built from a training input image.
/// @param inputImage The input grayscale image to segment.
/// @param inputTrainingImage The input label training image (16 or 32 bits) where each label represents a class sample for the training step.
/// @param featureGroup The groups of textural features to compute. This list defines all the textural attributes proposed for performing the classification.
/// @param radiusRange The minimum and maximum radius, in voxels, of the circular neighborhoods used for computing textural features.
/// @param radiusStep The step, in voxels, used to define the set of radius between minimum and maximum. The maximum radius is systematically added to the radius list.
/// @param coocRadius The radius, in voxels, of the circular neighborhood used by the co-occurrence features. This parameter is ignored if none of the co-occurrence feature groups is selected.
/// @param coocTextonShape The shape of the co-occurrence texton (the pattern defined by the set of co-occurrence vectors). This parameter is ignored if none of the co-occurrence feature groups is selected.
/// @param coocTextonSize The size, in voxels, of the texton shape for co-occurrence features. This parameter is ignored if none of the co-occurrence feature groups is selected.
/// @param minSeparationPercentage This parameter controls the rejection criteria of the feature selection algorithm (FS).
/// @param outputMapType The type of uncertainty map image to compute.
/// @param outputLabelImage The output label image representing the texture classification result. Its dimensions and type are forced to the same values as the training input.
/// @param outputMapImage The output map image. Its dimensions are forced to the same values as the training image. In CLASS_DISTANCE mode, its number of channels is equal to the number of classes defined in the training image. Its data type is forced to floating point.
/// @return Returns a SupervisedTextureClassification3dOutput structure containing the outputLabelImage and outputMapImage output parameters.
IMAGEDEV_CPP_API 
SupervisedTextureClassification3dOutput
supervisedTextureClassification3d( std::shared_ptr< iolink::ImageView > inputImage,
                                   std::shared_ptr< iolink::ImageView > inputTrainingImage,
                                   int32_t featureGroup,
                                   const iolink::Vector2u32& radiusRange,
                                   uint32_t radiusStep,
                                   uint32_t coocRadius,
                                   SupervisedTextureClassification3d::CoocTextonShape coocTextonShape,
                                   uint32_t coocTextonSize,
                                   double minSeparationPercentage,
                                   SupervisedTextureClassification3d::OutputMapType outputMapType,
                                   std::shared_ptr< iolink::ImageView > outputLabelImage = nullptr,
                                   std::shared_ptr< iolink::ImageView > outputMapImage = nullptr );
} // namespace imagedev
