/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <iolink/view/ImageView.h>

namespace imagedev
{
// Output structure of the volumeProjection3d function.
struct VolumeProjection3dOutput
{
    /// The output image representing the volume projection according to the selected criterion.
    std::shared_ptr< iolink::ImageView > outputImage;
    /// Output image indicating the selected plane in INTENSITY_MINIMUM or INTENSITY_MAXIMUM mode (not available in INTENSITY_MEAN mode).
    std::shared_ptr< iolink::ImageView > outputMapImage;
};

/// Creates a single image containing selected pixels from a volume image.
class IMAGEDEV_CPP_API VolumeProjection3d final : public GenericAlgorithm
{
public:
    /// The axis along which slices are projected.
    enum AxisDirection
    {
    /// The YZ slices are projected.
        X_AXIS = 0,
    /// The XZ slices are projected.
        Y_AXIS,
    /// The XY slices are projected.
        Z_AXIS
    };
    /// The projection criterion (the rule used to transform a stack of pixels into a single pixel).
    enum ProjectionMode
    {
    /// The pixel of highest intensity is selected in the stack. The map output is available with this mode.
        INTENSITY_MAXIMA = 0,
    /// The pixel of lowest intensity is selected in the stack. The map output is available with this mode.
        INTENSITY_MINIMA,
    /// The pixels of the stack are averaged along the selected axis. The map output is not available with this mode.
        INTENSITY_MEAN
    };
    /// Define the dimensional space of the output image.
    enum DimensionalSpace
    {
    /// The output image is a 2D image.
        SPACE_2D = 0,
    /// The output image is a 3D image.
        SPACE_3D
    };

    // Command constructor.
    VolumeProjection3d();


    /// Gets the inputImage parameter.
    /// The 3D input image.
    std::shared_ptr< iolink::ImageView > inputImage() const;
    /// Sets the inputImage parameter.
    /// The 3D input image.
    void setInputImage( std::shared_ptr< iolink::ImageView > inputImage );

    /// Gets the axisDirection parameter.
    /// The axis along which slices are projected.
    VolumeProjection3d::AxisDirection axisDirection() const;
    /// Sets the axisDirection parameter.
    /// The axis along which slices are projected.
    void setAxisDirection( const VolumeProjection3d::AxisDirection& axisDirection );

    /// Gets the projectionMode parameter.
    /// The projection criterion (the rule used to transform a stack of pixels into a single pixel).
    VolumeProjection3d::ProjectionMode projectionMode() const;
    /// Sets the projectionMode parameter.
    /// The projection criterion (the rule used to transform a stack of pixels into a single pixel).
    void setProjectionMode( const VolumeProjection3d::ProjectionMode& projectionMode );

    /// Gets the dimensionalSpace parameter.
    /// Define the dimensional space of the output image.
    VolumeProjection3d::DimensionalSpace dimensionalSpace() const;
    /// Sets the dimensionalSpace parameter.
    /// Define the dimensional space of the output image.
    void setDimensionalSpace( const VolumeProjection3d::DimensionalSpace& dimensionalSpace );

    /// Gets the createMapImage parameter.
    /// This parameter indicates whether the algorithm creates an output map image.
    bool createMapImage() const;
    /// Sets the createMapImage parameter.
    /// This parameter indicates whether the algorithm creates an output map image.
    void setCreateMapImage( const bool& createMapImage );

    /// Gets the outputImage parameter.
    /// The output image representing the volume projection according to the selected criterion.
    std::shared_ptr< iolink::ImageView > outputImage() const;
    /// Sets the outputImage parameter.
    /// The output image representing the volume projection according to the selected criterion.
    void setOutputImage( std::shared_ptr< iolink::ImageView > outputImage );

    /// Gets the outputMapImage parameter.
    /// Output image indicating the selected plane in INTENSITY_MINIMUM or INTENSITY_MAXIMUM mode (not available in INTENSITY_MEAN mode).
    std::shared_ptr< iolink::ImageView > outputMapImage() const;
    /// Sets the outputMapImage parameter.
    /// Output image indicating the selected plane in INTENSITY_MINIMUM or INTENSITY_MAXIMUM mode (not available in INTENSITY_MEAN mode).
    void setOutputMapImage( std::shared_ptr< iolink::ImageView > outputMapImage );

    // Method to launch the command.
    void execute();

};

/// Creates a single image containing selected pixels from a volume image.
/// @param inputImage The 3D input image.
/// @param axisDirection The axis along which slices are projected.
/// @param projectionMode The projection criterion (the rule used to transform a stack of pixels into a single pixel).
/// @param dimensionalSpace Define the dimensional space of the output image.
/// @param createMapImage This parameter indicates whether the algorithm creates an output map image.
/// @param outputImage The output image representing the volume projection according to the selected criterion.
/// @param outputMapImage Output image indicating the selected plane in INTENSITY_MINIMUM or INTENSITY_MAXIMUM mode (not available in INTENSITY_MEAN mode).
/// @return Returns a VolumeProjection3dOutput structure containing the outputImage and outputMapImage output parameters.
IMAGEDEV_CPP_API 
VolumeProjection3dOutput
volumeProjection3d( std::shared_ptr< iolink::ImageView > inputImage,
                    VolumeProjection3d::AxisDirection axisDirection,
                    VolumeProjection3d::ProjectionMode projectionMode,
                    VolumeProjection3d::DimensionalSpace dimensionalSpace,
                    bool createMapImage,
                    std::shared_ptr< iolink::ImageView > outputImage = nullptr,
                    std::shared_ptr< iolink::ImageView > outputMapImage = nullptr );
} // namespace imagedev
