/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <iolink/Vector.h>
#include <iolink/view/ImageView.h>

namespace imagedev
{
/// Generates a three-dimensional binary image representing the region of acquisition of an electron tilted image series.
class IMAGEDEV_CPP_API WedgeMaskImage3d final : public GenericAlgorithm
{
public:
    /// The axis on which the scene has been tilted during the acquisition.
    enum TiltAxis
    {
    /// Images were tilted on the X axis.
        X_TILT = 0,
    /// Images were tilted on the Y axis.
        Y_TILT,
    /// Images were tilted on the X and Y axis.
        DUAL_TILT
    };

    // Command constructor.
    WedgeMaskImage3d();


    /// Gets the tiltAxis parameter.
    /// The axis on which the scene has been tilted during the acquisition.
    WedgeMaskImage3d::TiltAxis tiltAxis() const;
    /// Sets the tiltAxis parameter.
    /// The axis on which the scene has been tilted during the acquisition.
    void setTiltAxis( const WedgeMaskImage3d::TiltAxis& tiltAxis );

    /// Gets the shape parameter.
    /// The X, Y, and Z dimensions, in voxels, of the output image.
    iolink::Vector3u32 shape() const;
    /// Sets the shape parameter.
    /// The X, Y, and Z dimensions, in voxels, of the output image.
    void setShape( const iolink::Vector3u32& shape );

    /// Gets the tiltAngle parameter.
    /// The minimum and maximum tilt angles, in degrees, for masking images acquired by tilted tomography. These angles are generally [-70, 70] degrees for electron tomography systems. Supported values are [-90, 90].
    iolink::Vector2d tiltAngle() const;
    /// Sets the tiltAngle parameter.
    /// The minimum and maximum tilt angles, in degrees, for masking images acquired by tilted tomography. These angles are generally [-70, 70] degrees for electron tomography systems. Supported values are [-90, 90].
    void setTiltAngle( const iolink::Vector2d& tiltAngle );

    /// Gets the outputBinaryImage parameter.
    /// The output 3D binary image.
    std::shared_ptr< iolink::ImageView > outputBinaryImage() const;
    /// Sets the outputBinaryImage parameter.
    /// The output 3D binary image.
    void setOutputBinaryImage( std::shared_ptr< iolink::ImageView > outputBinaryImage );

    // Method to launch the command.
    void execute();

};

/// Generates a three-dimensional binary image representing the region of acquisition of an electron tilted image series.
/// @param tiltAxis The axis on which the scene has been tilted during the acquisition.
/// @param shape The X, Y, and Z dimensions, in voxels, of the output image.
/// @param tiltAngle The minimum and maximum tilt angles, in degrees, for masking images acquired by tilted tomography. These angles are generally [-70, 70] degrees for electron tomography systems. Supported values are [-90, 90].
/// @param outputBinaryImage The output 3D binary image.
/// @return Returns the outputBinaryImage output parameter.
IMAGEDEV_CPP_API 
std::shared_ptr< iolink::ImageView >
wedgeMaskImage3d( WedgeMaskImage3d::TiltAxis tiltAxis,
                  const iolink::Vector3u32& shape,
                  const iolink::Vector2d& tiltAngle,
                  std::shared_ptr< iolink::ImageView > outputBinaryImage = nullptr );
} // namespace imagedev
