// ================================================================================ //
//       THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),        //
//                  AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                   //
//                                                                                  //
//     REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS      //
//     SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR      //
//                       WRITTEN AUTHORIZATION OF FEI S.A.S.                        //
//                                                                                  //
//                             RESTRICTED RIGHTS LEGEND                             //
//     USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS     //
//     WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN     //
//     SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT     //
//     CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN     //
//        TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.         //
//                                                                                  //
//                         COPYRIGHT (C) 2021 BY FEI S.A.S,                         //
//                                 BORDEAUX, FRANCE                                 //
//                               ALL RIGHTS RESERVED                                //
//                                                                                  //
//        SEE https://developer.openinventor.com/MiscFiles/EULA.pdf FOR MORE        //
// ================================================================================ //

#pragma once

#include <iolink/RegionX.h>
#include <iolink/Vector.h>

namespace iolink
{
/**
 * A multi-dimensional region defined by its origin and its size.
 *
 * Origin is always the left-bottom corner of the region.
 *
 * @tparam T The type of region indexers
 * @tparam N The dimension of the region.
 *
 * Region definition is different according to its indexer type.
 * Float-indexed-regions are defined from origin point (Org) to Max = Org + region size.
 * Integer-indexed-regions are defined from Org to Max = Org + region size - 1
 *
 * Up-boundaries are not included for integer-indexers-regions (except when size is 0
 * and region is a point).
 *
 * Up-boundaries are included for float-indexers-regions.
 *
 * Aliases can be used to made this class more practical to use. They use the
 * following convention:
 *    RegionNX
 *
 * Where N is the dimension and X the type. The types currently available are:
 * - F for single precision floats.
 * - D for double precision floats.
 * - U64 for uint64_t.
 *
 * Examples:
 * - Region2u64: 2 dimensional uint64_t region.
 */

template <typename T, size_t N>
class Region final
{
public:
  using VectorType = Vector<T, N>;
  using SizeType = Vector<T, N>;

  /**
   * Create an empty region
   */
  Region();

  /**
   * Create a region with given origin and size
   * @param origin vector containing region origin
   * @param size vector containing region size
   */
  Region(const VectorType& origin, const SizeType& size);

  /**
   * Build a region from another of different dimension and different type
   *
   * If dimension of other is smaller than dimension of region (ex. creating
   * a 3D region from a 2D region) behavior is as follow:
   * - For Integer-indexed-regions, origin is set to 0 and size is set to 1.
   * i.e, if creating a 3D volume from a 2D slice, resulting region will be ({x, y, 0}, {sizeX, sizeY, 1})
   * - For Float-indexed-regions, origin is set to 0 and size is set to 0.
   * i.e, if creating a 3D volume from a 2D slice, resulting region will be ({x, y, 0}, {sizeX, sizeY, 0})
   */
  template <typename U, size_t P>
  explicit Region(const Region<U, P>& other)
      : m_origin(other.min(), 0)
      , m_size(other.size(), 1)
  {
  }

  /**
   * Return number of dimension of this region
   */
  inline size_t dimensionCount() const { return N; }

  /**
   * Return region's origin
   */
  inline const VectorType& origin() const { return m_origin; }

  /**
   * return size of region
   * @return size vector of the region
   */
  inline const SizeType& size() const { return m_size; }

  /**
   * Return number of elements (pixel, voxel, samples) contained in the region.
   * For Integer-indexed-regions, it return product of each dimensions.
   * This method has no sense for Float-indexed-regions and thus return 0
   */
  size_t elementCount() const;

  /**
   * return min region coordinates
   * @return min region vector
   */
  inline VectorType min() const { return m_origin; }

  /**
   * return max region coordinates
   * @return max region vector
   */
  VectorType max() const;

  /**
   * Indicates if region is empty
   * @return if region is empty
   */
  bool isEmpty() const;

  /**
   * Returns if the current region contains the given one
   * @param other region to test if it is contained into current one
   * @return if the given region is contained into current one
   */
  bool contains(const Region& other) const;

  /**
   * Returns if the given region contains the given position vector
   * @param position point to test if it is contained into current one
   * @return if the given position is contained into current region
   */
  bool contains(const VectorType& position) const;

  /**
   * Returns if the given region intersects the given position vector
   * @param other region to test if it intersects with current one
   * @return if the given region intersects current region
   */
  bool intersect(const Region& other) const;

  /**
   * Returns the intersection region between current and given one
   * @param other region
   * @return intersection region if it intersects or empty region otherwise
   */
  Region intersection(const Region& other) const;

  /**
   * Returns a region of size one along one axis
   *
   * @param dimension axis on which the slice must be selected
   * @param origin position of slice in given dimension
   *
   * @return the slice at given position for given dimension
   * @throw InvalidArgument if dimension or origin are out of range
   *
   * e.g. a 3D region {{0, 0, 0}, {5, 6, 7}}
   * - extractSliceFromAxis(0, 3) returns 3D region {{3, 0, 0}, {1, 6, 7}}
   * - extractSliceFromAxis(1, 4) returns 3D region {{0, 4, 0}, {5, 1, 7}}
   * - extractSliceFromAxis(2, 5) returns 3D region {{0, 0, 5}, {5, 6, 1}}
   *
   * e.g. a 3D region with non-null origin { {2, 3, 4}, {5, 6, 7}}
   *
   * extractSliceFromAxis(0, 3) returns 3D region {{5, 3, 4}, {1, 6, 7}}
   */
  Region extractSliceFromAxis(size_t dimension, T origin) const;

  /**
   * Returns the intersection region between current and given one
   * @param other region
   * @return if current region and given one are identical
   * (same origin if both null size, same origin and size otherwise)
   */
  bool operator==(const Region& other) const;

  Region(const RegionX<T>& other);

  operator RegionX<T>() const;

  /**
   * Return a string representation
   */
  std::string toString() const;

private:
  VectorType m_origin;

  // size computation depends on given indexer type T.
  // For integer indexers, size is equal to max()- min() + 1
  // For float indexers, size is equal to max()- min()
  SizeType m_size;
};

// ============================ //
// Extern template declarations //
// ============================ //

extern template class IOLINK_API_IMPORT Region<uint64_t, 2>;

extern template class IOLINK_API_IMPORT Region<uint64_t, 3>;

extern template class IOLINK_API_IMPORT Region<uint64_t, 4>;

// ================= //
// Alias definitions //
// ================= //

template <typename T>
using Region2 = Region<T, 2>;
template <typename T>
using Region3 = Region<T, 3>;
template <typename T>
using Region4 = Region<T, 4>;

using Region2u64 = Region2<uint64_t>; ///< Alias for uint64_t 2D regions
using Region3u64 = Region3<uint64_t>; ///< Alias for uint64_t 3D regions
using Region4u64 = Region4<uint64_t>; ///< Alias for uint64_t 4D regions

} // end namespace iolink
