// ================================================================================ //
//       THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),        //
//                  AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                   //
//                                                                                  //
//     REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS      //
//     SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR      //
//                       WRITTEN AUTHORIZATION OF FEI S.A.S.                        //
//                                                                                  //
//                             RESTRICTED RIGHTS LEGEND                             //
//     USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS     //
//     WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN     //
//     SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT     //
//     CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN     //
//        TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.         //
//                                                                                  //
//                         COPYRIGHT (C) 2021 BY FEI S.A.S,                         //
//                                 BORDEAUX, FRANCE                                 //
//                               ALL RIGHTS RESERVED                                //
//                                                                                  //
//        SEE https://developer.openinventor.com/MiscFiles/EULA.pdf FOR MORE        //
// ================================================================================ //

#pragma once

#include <iolink/IOLinkAPI.h>
#include <iolink/Indexer.h>
#include <iolink/Region.h>
#include <iolink/Vector.h>
#include <iolink/storage/RandomAccess.h>
#include <memory>

namespace iolink
{

class InternalRegionCopier;

/**
 * This class contains the algorithms used to copy data beetween two regions
 * of N dimensionnal data.
 *
 * @verbatim
 *     SOURCE        DESTINATION
 *  -------------     --------
 * |      o----  |   | o----  |
 * |      |  --|-|---|-|->  | |
 * |      |    | |   | |    | |
 * |       ----  |   |  ----  |
 * |             |    --------
 * |             |
 *  -------------
 * @endverbatim
 *
 * The methods of this class will copy one region in the source, to another
 * in the destination. With some conditions: the source region and the
 * destination one must have exactly the same size, and the source and the
 * destination must share the same DataType.
 */
class IOLINK_API RegionCopier final
{
public:
  /**
   * Build a copier with all the parameters.
   *
   * If one or both of the input indexers have an unknown data type, a fallback
   * mechanism will be trigerred, using an equivalent version with the data type
   * set to the one given in parameter.
   *
   * @param srcIndexer An indexer used to access source
   * @param dstIndexer An indexer used to access destination
   * @param srcOrigin Position where the first sample will be read
   * @param dstOrigin Position where the first sample will be written
   * @param regionSize The size of the region to copy
   * @param dataType The type of data stored in the source and the destination
   *
   * @throw InvalidArgument If the indexers data type is not the same as dataType.
   */
  RegionCopier(const Indexer& srcIndexer,
               const Indexer& dstIndexer,
               const VectorXu64& srcOrigin,
               const VectorXu64& dstOrigin,
               const VectorXu64& regionSize,
               DataType dataType);

  /**
   * Shortened constructor in the case if the destination has the same
   * dimensions as the region itself.
   *
   * If the input indexer has an unknown data type, a fallback mecanism will be
   * triggered, using an equivalent version with the data type set to the one
   * given in parameter.
   *
   * Used for example when reading a part of an image into a buffer.
   * @param srcindexer  An indexer used to access source
   * @param srcRegion region to copy from the source
   * @param dataType datatype of source data
   *
   * @throw InvalidArgument If the srcIndexer data type is not the same as dataType.
   */
  RegionCopier(const Indexer& srcindexer, const RegionXu64& srcRegion, DataType dataType);

  /**
   * Shortened constructor in the case if the source has the same
   * dimensions as the region itself.
   *
   * If the input indexer has an unknown data type, a fallback mecanism will be
   * triggered, using an equivalent version with the data type set to the one
   * given in parameter.
   *
   * Used for example to write data stored in a buffer into a part of an image.
   * @param dstRegion region destination to fill
   * @param dstIndexer An indexer used to access destination
   * @param dataType datatype of data
   *
   * @throw InvalidArgument If the dstIndexer data type is not the same as dataType.
   */
  RegionCopier(const RegionXu64& dstRegion, const Indexer& dstIndexer, DataType dataType);

  /**
   * Copy constructor
   * @param other RegionCopier to use as model
   */
  RegionCopier(const RegionCopier& other);

  /**
   * Assignation operator
   * @param other RegionCopier to use for assignment
   */
  RegionCopier& operator=(const RegionCopier& other);

  /**
   * Destructor
   */
  ~RegionCopier();

  /**
   * Copy data from memory, to memory.
   *
   * @param src A pointer to the data to copy
   * @param dst A pointer where to copy the data
   */
  void copy(const uint8_t* src, uint8_t* dst);

  /**
   * Copy data from a storage to memory.
   *
   * @param src A RandomAccess used to read data
   * @param dst A pointer where to copy the data
   */
  void copy(std::shared_ptr<RandomAccess> src, uint8_t* dst);

  /**
   * Copy data from memory to a storage.
   *
   * @param src A pointer to the data to copy
   * @param dst A RandomAccess used to write data
   */
  void copy(const uint8_t* src, std::shared_ptr<RandomAccess> dst);

private:
  InternalRegionCopier* m_impl;
};

} // end namespace iolink
