// ================================================================================ //
//       THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),        //
//                  AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                   //
//                                                                                  //
//     REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS      //
//     SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR      //
//                       WRITTEN AUTHORIZATION OF FEI S.A.S.                        //
//                                                                                  //
//                             RESTRICTED RIGHTS LEGEND                             //
//     USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS     //
//     WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN     //
//     SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT     //
//     CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN     //
//        TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.         //
//                                                                                  //
//                         COPYRIGHT (C) 2021 BY FEI S.A.S,                         //
//                                 BORDEAUX, FRANCE                                 //
//                               ALL RIGHTS RESERVED                                //
//                                                                                  //
//        SEE https://developer.openinventor.com/MiscFiles/EULA.pdf FOR MORE        //
// ================================================================================ //

#pragma once

#include <iolink/IOLinkAPI.h>
#include <iolink/VectorX.h>

namespace iolink
{

/**
 * A Region using dynamic vectors
 */
template <typename T>
class RegionX final
{
public:
  using VectorType = VectorX<T>;
  using SizeType = VectorX<T>;

  static RegionX fromIndex(const VectorX<T>& index);

  /**
   * Creates an empty region.
   *
   * @param dimension The dimension of the region.
   */
  explicit RegionX(size_t dimension);

  /** copy constructor */
  RegionX(const RegionX& region) = default;
  RegionX& operator=(const RegionX& other) = default;

  /** move constructor */
  RegionX(RegionX&& region) noexcept = default;
  RegionX& operator=(RegionX&& other) noexcept = default;

  /**
   * Builds a region from its origin and size.
   */
  RegionX(const VectorType& origin, const SizeType& size);

  /**
   * Utility factory that create the region of origin [0, 0, 0] and given size.
   */
  static RegionX createFullRegion(const SizeType& regionSize);

  /**
   * The dimension of this region.
   */
  size_t dimensionCount() const;

  /**
   * Returns the position of the region's origin.
   */
  const VectorType& origin() const;

  /**
   * Returns the region's size.
   */
  const SizeType& size() const;

  /**
   * Returns the minimal point of the region.
   *
   * Alias for origin.
   */
  VectorType min() const;

  /**
   * Return the maximal point of the region.
   */
  VectorType max() const;

  /**
   * Returns the number of cells in this region.
   *
   * Only applicable to integer typed regions.
   * It will return 0 for floating point regions.
   */
  size_t elementCount() const;

  /**
   * Checks if the region is empty.
   *
   * An integer typed region is empty if one of its sizes is 0.
   * A floating point region is empty is one of its sizes is Nan.
   */
  bool isEmpty() const;

  /**
   * Checks if this region conatins another.
   * @throw InvalidArgument if region don't have compatible dimensions
   */
  bool contains(const RegionX& other) const;

  /**
   * Checks if this region contains a point.
   * @throw InvalidArgument if region don't have compatible dimensions
   */
  bool contains(const VectorType& position) const;

  /**
   * Checks if this region intersect another.
   * @throw InvalidArgument if region don't have compatible dimensions
   */
  bool intersect(const RegionX& other) const;

  /**
   * Returns the intersection of this region and another.
   *
   * If regions does not intersect, it returns an empty region.
   * @throw InvalidArgument if region don't have compatible dimensions
   */
  RegionX intersection(const RegionX& other) const;

  /**
   * Returns a region of size one along one axis
   *
   * @param dimension axis on which the slice must be selected
   * @param origin position of slice in given dimension
   *
   * @return the slice at given position for given dimension
   * @throw InvalidArgument if dimension or origin are out of range
   *
   * e.g. a 3D region {{0, 0, 0}, {5, 6, 7}}
   * - extractSliceFromAxis(0, 3) returns 3D region {{3, 0, 0}, {1, 6, 7}}
   * - extractSliceFromAxis(1, 4) returns 3D region {{0, 4, 0}, {5, 1, 7}}
   * - extractSliceFromAxis(2, 5) returns 3D region {{0, 0, 5}, {5, 6, 1}}
   *
   * e.g. a 3D region with non-null origin { {2, 3, 4}, {5, 6, 7}}
   *
   * extractSliceFromAxis(0, 3) returns 3D region {{5, 3, 4}, {1, 6, 7}}
   */
  RegionX extractSliceFromAxis(size_t dimension, T origin) const;

  bool operator==(const RegionX& other) const;
  bool operator!=(const RegionX& other) const;

  /**
   * Return a string representation
   */
  std::string toString() const;

private:
  VectorType m_origin;
  SizeType m_size;
};

// ============================ //
// Extern template declarations //
// ============================ //

extern template class IOLINK_API_IMPORT RegionX<uint64_t>;

// ================= //
// Alias definitions //
// ================= //

using RegionXu64 = RegionX<uint64_t>; ///< Alias for uint64_t variable size regions

} // end namespace iolink
