// ================================================================================ //
//       THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),        //
//                  AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                   //
//                                                                                  //
//     REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS      //
//     SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR      //
//                       WRITTEN AUTHORIZATION OF FEI S.A.S.                        //
//                                                                                  //
//                             RESTRICTED RIGHTS LEGEND                             //
//     USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS     //
//     WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN     //
//     SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT     //
//     CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN     //
//        TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.         //
//                                                                                  //
//                         COPYRIGHT (C) 2021 BY FEI S.A.S,                         //
//                                 BORDEAUX, FRANCE                                 //
//                               ALL RIGHTS RESERVED                                //
//                                                                                  //
//        SEE https://developer.openinventor.com/MiscFiles/EULA.pdf FOR MORE        //
// ================================================================================ //

#pragma once

#include <iolink/IOLinkAPI.h>
#include <iolink/VariantDataValue.h>

namespace iolink
{
/**
 * Converter contains a set of method allowing to interpret a VariantDataValue as a determined type
 */
class IOLINK_API VariantDataValueConverter
{
public:
  /** Return converted metadata value into 8-bit signed integers
   *
   * @param value VariantDataValue to convert
   * @throw Error if VariantDataValue original type cannot be converted
   */
  static int8_t toInt8(std::shared_ptr<VariantDataValue> value);

  /** Return converted metadata value into 16-bit signed integers
   *
   * @param value VariantDataValue to convert
   * @throw Error if VariantDataValue original type cannot be converted
   */
  static int16_t toInt16(std::shared_ptr<VariantDataValue> value);

  /** Return converted metadata value into 32-bit signed integers
   *
   * @param value VariantDataValue to convert
   * @throw Error if VariantDataValue original type cannot be converted
   */
  static int32_t toInt32(std::shared_ptr<VariantDataValue> value);

  /** Return converted metadata value into 64-bit signed integers
   *
   * @param value VariantDataValue to convert
   * @throw Error if VariantDataValue original type cannot be converted
   */
  static int64_t toInt64(std::shared_ptr<VariantDataValue> value);

  /** Return converted metadata value into 8-bit unsigned integers
   *
   * @param value VariantDataValue to convert
   * @throw Error if VariantDataValue original type cannot be converted
   */
  static uint8_t toUint8(std::shared_ptr<VariantDataValue> value);

  /** Return converted metadata value into 16-bit unsigned integers
   *
   * @param value VariantDataValue to convert
   * @throw Error if VariantDataValue original type cannot be converted
   */
  static uint16_t toUint16(std::shared_ptr<VariantDataValue> value);

  /** Return converted metadata value into 32-bit unsigned integers
   *
   * @param value VariantDataValue to convert
   * @throw Error if VariantDataValue original type cannot be converted
   */
  static uint32_t toUint32(std::shared_ptr<VariantDataValue> value);

  /** Return converted metadata value into 32-bit signed integers
   *
   * @param value VariantDataValue to convert
   * @throw Error if VariantDataValue original type cannot be converted
   */
  static uint64_t toUint64(std::shared_ptr<VariantDataValue> value);

  /** Return converted metadata value into 32-bit floating-point
   *
   * @param value VariantDataValue to convert
   * @throw Error if VariantDataValue original type cannot be converted
   */
  static float toFloat(std::shared_ptr<VariantDataValue> value);

  /** Return converted metadata value into 64-bit floats
   *
   * @param value VariantDataValue to convert
   * @throw Error if VariantDataValue original type cannot be converted
   */
  static double toDouble(std::shared_ptr<VariantDataValue> value);

  /** Return converted metadata value into string
   *
   * @param value VariantDataValue to convert
   * @throw Error if VariantDataValue original type cannot be converted
   */
  static std::string toString(std::shared_ptr<VariantDataValue> value);

  //===================================================================
  //======== Array extraction =========================================
  //===================================================================

  /** Convert VariantDataValues into 8-bit signed integers.
   *
   * @param variant VariantDataValue to convert
   * @param values Pre-allocated array which will be filled with converted values extracted
   * from VariantDataValue
   * @param count Size of pre-allocated array (should be at least equal to VariantDataValue count)
   * @return effective count of values converted into values
   * @throw Error if VariantDataValue original type cannot be converted
   */
  static size_t toInt8Array(std::shared_ptr<VariantDataValue> variant, int8_t* values, size_t count);

  /** Convert VariantDataValues into 16-bit signed integers
   *
   * @param variant VariantDataValue to convert
   * @param values Pre-allocated array which will be filled with converted values extracted
   * from VariantDataValue
   * @param count Size of pre-allocated array (should be at least equal to VariantDataValue count)
   * @return effective count of values converted into values
   * @throw Error if VariantDataValue original type cannot be converted
   */
  static size_t toInt16Array(std::shared_ptr<VariantDataValue> variant, int16_t* values, size_t count);

  /** Convert VariantDataValues into 32-bit signed integers
   *
   * @param variant VariantDataValue to convert
   * @param values Pre-allocated array which will be filled with converted values extracted
   * from VariantDataValue
   * @param count Size of pre-allocated array (should be at least equal to VariantDataValue count)
   * @return effective count of values converted into values
   * @throw Error if VariantDataValue original type cannot be converted
   */
  static size_t toInt32Array(std::shared_ptr<VariantDataValue> variant, int32_t* values, size_t count);

  /** Convert VariantDataValues into 64-bit signed integers
   *
   * @param variant VariantDataValue to convert
   * @param values Pre-allocated array which will be filled with converted values extracted
   * from VariantDataValue
   * @param count Size of pre-allocated array (should be at least equal to VariantDataValue count)
   * @return effective count of values converted into values
   * @throw Error if VariantDataValue original type cannot be converted
   */
  static size_t toInt64Array(std::shared_ptr<VariantDataValue> variant, int64_t* values, size_t count);

  /** Convert VariantDataValues into 8-bit unsigned integers
   *
   * @param variant VariantDataValue to convert
   * @param values Pre-allocated array which will be filled with converted values extracted
   * from VariantDataValue
   * @param count Size of pre-allocated array (should be at least equal to VariantDataValue count)
   * @return effective count of values converted into values
   * @throw Error if VariantDataValue original type cannot be converted
   */
  static size_t toUint8Array(std::shared_ptr<VariantDataValue> variant, uint8_t* values, size_t count);

  /** Convert VariantDataValues into 16-bit unsigned integers
   *
   * @param variant VariantDataValue to convert
   * @param values Pre-allocated array which will be filled with converted values extracted
   * from VariantDataValue
   * @param count Size of pre-allocated array (should be at least equal to VariantDataValue count)
   * @return effective count of values converted into values
   * @throw Error if VariantDataValue original type cannot be converted
   */
  static size_t toUint16Array(std::shared_ptr<VariantDataValue> variant, uint16_t* values, size_t count);

  /** Convert VariantDataValues into 32-bit unsigned integers
   *
   * @param variant VariantDataValue to convert
   * @param values Pre-allocated array which will be filled with converted values extracted
   * from VariantDataValue
   * @param count Size of pre-allocated array (should be at least equal to VariantDataValue count)
   * @return effective count of values converted into values
   * @throw Error if VariantDataValue original type cannot be converted
   */
  static size_t toUint32Array(std::shared_ptr<VariantDataValue> variant, uint32_t* values, size_t count);

  /** Convert VariantDataValues into 64-bit signed integers
   *
   * @param variant VariantDataValue to convert
   * @param values Pre-allocated array which will be filled with converted values extracted
   * from VariantDataValue
   * @param count Size of pre-allocated array (should be at least equal to VariantDataValue count)
   * @return effective count of values converted into values
   * @throw Error if VariantDataValue original type cannot be converted
   */
  static size_t toUint64Array(std::shared_ptr<VariantDataValue> variant, uint64_t* values, size_t count);

  /** Convert VariantDataValues into 32-bit floating-point
   *
   * @param variant VariantDataValue to convert
   * @param values Pre-allocated array which will be filled with converted values extracted
   * from VariantDataValue
   * @param count Size of pre-allocated array (should be at least equal to VariantDataValue count)
   * @return effective count of values converted into values
   * @throw Error if VariantDataValue original type cannot be converted
   */
  static size_t toFloatArray(std::shared_ptr<VariantDataValue> variant, float* values, size_t count);

  /** Convert VariantDataValues into 64-bit floats
   *
   * @param variant VariantDataValue to convert
   * @param values Pre-allocated array which will be filled with converted values extracted
   * from VariantDataValue
   * @param count Size of pre-allocated array (should be at least equal to VariantDataValue count)
   * @return effective count of values converted into values
   * @throw Error if VariantDataValue original type cannot be converted
   */
  static size_t toDoubleArray(std::shared_ptr<VariantDataValue> variant, double* values, size_t count);

  /** Convert VariantDataValues into string
   *
   * @param variant VariantDataValue to convert
   * @param values Pre-allocated array which will be filled with converted values extracted
   * from VariantDataValue
   * @param count Size of pre-allocated array (should be at least equal to VariantDataValue count)
   * @return effective count of values converted into values
   * @throw Error if VariantDataValue original type cannot be converted
   */
  static size_t toStringArray(std::shared_ptr<VariantDataValue> variant, std::string* values, size_t count);

  //===================================================================
  //======== Nth value of array extraction ===========================
  //===================================================================

  /** Convert VariantDataValue at given index into 8-bit signed integers.
   *
   * @param variant VariantDataValue to convert
   * @param index Index of the value to convert and return from VariantDataValue
   * @throw Error if VariantDataValue original type cannot be converted
   * @throw Error if given index is out of range
   * @return the converted value of VariantDataValue at given index
   */
  static int8_t toInt8(std::shared_ptr<VariantDataValue> variant, size_t index);

  /** Convert VariantDataValue at given index into 16-bit signed integers.
   *
   * @param variant VariantDataValue to convert
   * @param index Index of the value to convert and return from VariantDataValue
   * @throw Error if VariantDataValue original type cannot be converted
   * @throw Error if given index is out of range
   * @return the converted value of VariantDataValue at given index
   */
  static int16_t toInt16(std::shared_ptr<VariantDataValue> variant, size_t index);

  /** Convert VariantDataValue at given index into 32-bit signed integers.
   *
   * @param variant VariantDataValue to convert
   * @param index Index of the value to convert and return from VariantDataValue
   * @throw Error if VariantDataValue original type cannot be converted
   * @throw Error if given index is out of range
   * @return the converted value of VariantDataValue at given index
   */
  static int32_t toInt32(std::shared_ptr<VariantDataValue> variant, size_t index);

  /** Convert VariantDataValue at given index into 64-bit signed integers.
   *
   * @param variant VariantDataValue to convert
   * @param index Index of the value to convert and return from VariantDataValue
   * @throw Error if VariantDataValue original type cannot be converted
   * @throw Error if given index is out of range
   * @return the converted value of VariantDataValue at given index
   */
  static int64_t toInt64(std::shared_ptr<VariantDataValue> variant, size_t index);

  /** Convert VariantDataValue at given index into 8-bit unsigned integers.
   *
   * @param variant VariantDataValue to convert
   * @param index Index of the value to convert and return from VariantDataValue
   * @throw Error if VariantDataValue original type cannot be converted
   * @throw Error if given index is out of range
   * @return the converted value of VariantDataValue at given index
   */
  static uint8_t toUint8(std::shared_ptr<VariantDataValue> variant, size_t index);

  /** Convert VariantDataValue at given index into 16-bit unsigned integers.
   *
   * @param variant VariantDataValue to convert
   * @param index Index of the value to convert and return from VariantDataValue
   * @throw Error if VariantDataValue original type cannot be converted
   * @throw Error if given index is out of range
   * @return the converted value of VariantDataValue at given index
   */
  static uint16_t toUint16(std::shared_ptr<VariantDataValue> variant, size_t index);

  /** Convert VariantDataValue at given index into 32-bit unsigned integers.
   *
   * @param variant VariantDataValue to convert
   * @param index Index of the value to convert and return from VariantDataValue
   * @throw Error if VariantDataValue original type cannot be converted
   * @throw Error if given index is out of range
   * @return the converted value of VariantDataValue at given index
   */
  static uint32_t toUint32(std::shared_ptr<VariantDataValue> variant, size_t index);

  /** Convert VariantDataValue at given index into 64-bit unsigned integers.
   *
   * @param variant VariantDataValue to convert
   * @param index Index of the value to convert and return from VariantDataValue
   * @throw Error if VariantDataValue original type cannot be converted
   * @throw Error if given index is out of range
   * @return the converted value of VariantDataValue at given index
   */
  static uint64_t toUint64(std::shared_ptr<VariantDataValue> variant, size_t index);

  /** Convert VariantDataValue at given index into 32-bit floating-point.
   *
   * @param variant VariantDataValue to convert
   * @param index Index of the value to convert and return from VariantDataValue
   * @throw Error if VariantDataValue original type cannot be converted
   * @throw Error if given index is out of range
   * @return the converted value of VariantDataValue at given index
   */
  static float toFloat(std::shared_ptr<VariantDataValue> variant, size_t index);

  /** Convert VariantDataValue at given index into 64-bit floats.
   *
   * @param variant VariantDataValue to convert
   * @param index Index of the value to convert and return from VariantDataValue
   * @throw Error if VariantDataValue original type cannot be converted
   * @throw Error if given index is out of range
   * @return the converted value of VariantDataValue at given index
   */
  static double toDouble(std::shared_ptr<VariantDataValue> variant, size_t index);

  /** Convert VariantDataValue at given index into string.
   *
   * @param variant VariantDataValue to convert
   * @param index Index of the value to convert and return from VariantDataValue
   * @throw Error if VariantDataValue original type cannot be converted
   * @throw Error if given index is out of range
   * @return the converted value of VariantDataValue at given index
   */
  static std::string toString(std::shared_ptr<VariantDataValue> variant, size_t index);
};

} // end namespace iolink
