// ================================================================================ //
//       THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),        //
//                  AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                   //
//                                                                                  //
//     REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS      //
//     SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR      //
//                       WRITTEN AUTHORIZATION OF FEI S.A.S.                        //
//                                                                                  //
//                             RESTRICTED RIGHTS LEGEND                             //
//     USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS     //
//     WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN     //
//     SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT     //
//     CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN     //
//        TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.         //
//                                                                                  //
//                         COPYRIGHT (C) 2021 BY FEI S.A.S,                         //
//                                 BORDEAUX, FRANCE                                 //
//                               ALL RIGHTS RESERVED                                //
//                                                                                  //
//        SEE https://developer.openinventor.com/MiscFiles/EULA.pdf FOR MORE        //
// ================================================================================ //

#pragma once

#include <iolink/IOLinkAPI.h>

#include <array>
#include <cstdint>
#include <initializer_list>
#include <iostream>

namespace iolink
{

/**
 * A dynamically sized arithmetic vector.
 */
template <typename T>
class VectorX final
{
public:
  using ValueType = T;

  using Iterator = ValueType*;
  using ConstIterator = const ValueType*;

  /**
   * Creates a vector filled with one value.
   *
   * @param size The size of the vector.
   * @param value The value used to fill the vector.
   */
  static VectorX createUniform(size_t size, ValueType value);

  /**
   * Creates an empty vector.
   */
  VectorX();

  /**
   * Creates an unitialized vector.
   *
   * @param size The size of the vector.
   */
  explicit VectorX(size_t size);

  /**
   * Creates a vector from an initialization list.
   */
  VectorX(std::initializer_list<ValueType> init);

  /**
   * Build a vector from a vector storing another type.
   *
   * @tparam U The type of the source vector.
   * @param other The vector to convert.
   */
  template <typename U>
  explicit VectorX(const VectorX<U>& other)
      : VectorX(other.size())
  {
    for (size_t i = 0; i < m_size; ++i)
    {
      m_data[i] = static_cast<ValueType>(other[i]);
    }
  }

  VectorX(const VectorX& other);
  VectorX& operator=(const VectorX& other);

  VectorX(VectorX&& other) noexcept;
  VectorX& operator=(VectorX&& other) noexcept;

  ~VectorX();

  /**
   * Returns the size of the vector, which also correspond to vector number of dimension.
   */
  inline size_t size() const { return m_size; }

  /**
   * Access vector's value at given index.
   */
  inline ValueType at(size_t index) const { return m_data[index]; }

  /**
   * Set value as given index.
   */
  inline void setAt(size_t index, ValueType value) { m_data[index] = value; }

  inline ValueType& operator[](size_t index) { return m_data[index]; }
  inline ValueType operator[](size_t index) const { return m_data[index]; }

  inline Iterator begin() noexcept { return m_data; }
  inline ConstIterator begin() const noexcept { return m_data; }
  inline Iterator end() noexcept { return static_cast<Iterator>(m_data + m_size); }
  inline ConstIterator end() const noexcept { return static_cast<ConstIterator>(m_data + m_size); }

  /**
   * Direct access to the vector's data.
   */
  inline ValueType* data() { return m_data; }

  /**
   * Direct const access to the vector's data.
   */
  inline const ValueType* data() const { return m_data; }

  bool operator==(const VectorX& other) const;
  bool operator!=(const VectorX& other) const;

  /**
   * Returns the vector's norm, squared.
   *
   * Specifically, it is the L1 norm that's computed here.
   */
  double squaredLength() const;

  /**
   * Returns the vector's norm.
   *
   * Specifically, it is the L1 norm that's computed here.
   */
  double length() const;

  /**
   * Normalize the vector
   */
  void normalize();

  std::string toString() const;

  // ==================== //
  // Arithmetic operators //
  // ==================== //

  VectorX operator-() const;

  VectorX& operator+=(const VectorX& v);
  VectorX& operator-=(const VectorX& v);

  inline VectorX operator+(const VectorX& v) const { return VectorX(*this) += v; }
  inline VectorX operator-(const VectorX& v) const { return VectorX(*this) -= v; }

  VectorX& operator*=(ValueType value);
  VectorX& operator/=(ValueType value);

  inline VectorX operator*(ValueType s) const { return VectorX(*this) *= s; }
  inline VectorX operator/(ValueType s) const { return VectorX(*this) /= s; }

  friend inline VectorX operator*(ValueType s, VectorX v) { return v *= s; }

  /**
   * Dot product
   */
  ValueType dot(const VectorX& v) const;

  /**
   * Cross product of two vectors.
   *
   * Work only for vectors of size 3.
   */
  VectorX cross(const VectorX& v) const;

  // ==================== //
  // GLSL style operators //
  // ==================== //

  VectorX& operator*=(const VectorX& v);
  VectorX& operator/=(const VectorX& v);

  inline VectorX operator*(const VectorX& v) const { return VectorX(*this) *= v; }
  inline VectorX operator/(const VectorX& v) const { return VectorX(*this) /= v; }

  bool operator<(const VectorX& other) const;
  bool operator<=(const VectorX& other) const;
  bool operator>(const VectorX& other) const;
  bool operator>=(const VectorX& other) const;

private:
  size_t m_size;

  // Pointer on values. If m_size < m_array.size(), it point to array. Else,
  // it point to memory allocated on heap with new[].
  // In case m_data != m_array.data(), it should be deallocated with delete[]
  ValueType* m_data;

  // 4 sized array storing values if VectorX size is < 4.
  // This avoid allocating values on heap and greatly optimise VectorX in most common case.
  std::array<ValueType, 4> m_array;
};

template <typename T>
inline std::ostream&
operator<<(std::ostream& os, const VectorX<T>& v)
{
  os << v.toString();
  return os;
}

// ===================== //
// Template declarations //
// ===================== //

extern template class IOLINK_API_IMPORT VectorX<uint8_t>;
extern template class IOLINK_API_IMPORT VectorX<int8_t>;
extern template class IOLINK_API_IMPORT VectorX<uint16_t>;
extern template class IOLINK_API_IMPORT VectorX<int16_t>;
extern template class IOLINK_API_IMPORT VectorX<uint32_t>;
extern template class IOLINK_API_IMPORT VectorX<int32_t>;
extern template class IOLINK_API_IMPORT VectorX<uint64_t>;
extern template class IOLINK_API_IMPORT VectorX<int64_t>;
extern template class IOLINK_API_IMPORT VectorX<float>;
extern template class IOLINK_API_IMPORT VectorX<double>;

//==================== //
// Aliases declaration //
//==================== //

using VectorXu8 = VectorX<uint8_t>;   ///< Alias for uint8_t variable size vectors
using VectorXu16 = VectorX<uint16_t>; ///< Alias for uint16_t variable size vectors
using VectorXu32 = VectorX<uint32_t>; ///< Alias for uint32_t variable size vectors
using VectorXu64 = VectorX<uint64_t>; ///< Alias for uint64_t variable size vectors
using VectorXi8 = VectorX<int8_t>;    ///< Alias for int8_t variable size vectors
using VectorXi16 = VectorX<int16_t>;  ///< Alias for int16_t variable size vectors
using VectorXi32 = VectorX<int32_t>;  ///< Alias for int32_t variable size vectors
using VectorXi64 = VectorX<int64_t>;  ///< Alias for int64_t variable size vectors
using VectorXf = VectorX<float>;      ///< Alias for float variable size vectors
using VectorXd = VectorX<double>;     ///< Alias for double variable size vectors

} // end namespace iolink
