#pragma once

#include <memory>
#include <string>

#include <iolink/DataType.h>
#include <iolink/view/ImageView.h>

#include <iolink/cuda/api.h>

namespace iolink_cuda
{

/**
 * Factory to create Cuda ImageViews
 */
class IOLINK_CUDA_API CudaImageFactory
{

public:
  CudaImageFactory() = delete;
  CudaImageFactory(const CudaImageFactory& other) = delete;            // copy constructor
  CudaImageFactory(CudaImageFactory&& other) = delete;                 // move constructor
  CudaImageFactory& operator=(const CudaImageFactory& other) = delete; // copy assignment
  CudaImageFactory& operator=(CudaImageFactory&& other) = delete;      // move assignment

  /**
   * Allocates an image in CUDA memory
   *
   * @param shape Shape of the image to allocate
   * @param dtype DataType of the image to allocate
   */
  static std::shared_ptr<iolink::ImageView> allocate(const iolink::VectorXu64& shape, iolink::DataType dtype);

  /**
   * Allocates an image in CUDA memory
   *
   * @param shape Shape of the image to allocate
   * @param dtype iolink::DataType of the image to allocate
   * @param properties iolink::ImageProperties to apply to new image
   * @param metadata Metadata to apply to new image
   *
   * @throw Error if properties are null or invalid
   * @return an image allocated in CUDA memory
   */
  static std::shared_ptr<iolink::ImageView> allocate(const iolink::VectorXu64& shape,
                                                     iolink::DataType dtype,
                                                     std::shared_ptr<iolink::ImageProperties> properties,
                                                     std::shared_ptr<iolink::MetadataNode> metadata);

  /**
   * @brief Allocates an image in CUDA memory with coalesced memory layout.
   *
   * @param shape the shape of the image to allocate.
   * @param dtype The data type of the image to allocate.
   *
   * @return An ImageView allocated in CUDA memory with coalesced memory layout.
   */
  static std::shared_ptr<iolink::ImageView> allocateCoalesced(const iolink::VectorXu64& shape, iolink::DataType dtype);

  /**
   * @brief Allocates an image in CUDA memory with coalesced memory layout.
   *
   * @param shape the shape of the image to allocate.
   * @param dtype The data type of the image to allocate.
   * @param properties properties to apply to the new image.
   * @param metadata metadata to attach to the new image.
   *
   * @return An ImageView allocated in CUDA memory with coalesced memory layout.
   */
  static std::shared_ptr<iolink::ImageView> allocateCoalesced(const iolink::VectorXu64& shape,
                                                              iolink::DataType dtype,
                                                              std::shared_ptr<iolink::ImageProperties> properties,
                                                              std::shared_ptr<iolink::MetadataNode> metadata);

  /**
   * Copy given image in CUDA memory. If given image is already in CUDA memory, it
   * is returned directly.
   *
   * Cuda memory is allocated by this operation. Data from original ImageView
   * are duplicated.
   *
   * Returned ImageView has following capabilities (see #ImageCapability):
   * - READ
   * - WRITE
   *
   * @param image Image to load
   *
   * @throw InvalidArgument If the input image is null
   * @throw Error If the input ImageView does not support READ capability
   * @throw InvalidArgument If properties of input ImageView are not valid
   */
  static std::shared_ptr<iolink::ImageView> copyInCudaMemory(std::shared_ptr<iolink::ImageView> image);

  /**
   * Adapt any ImageView to accept CUDA buffers for read and write operations.
   * If given image is already a CUDA image, it is returned directly.
   *
   * @param image Image to adapt for Cuda buffers
   * @throw InvalidArgument If properties of input ImageView are not valid
   */
  static std::shared_ptr<iolink::ImageView> adaptToCuda(std::shared_ptr<iolink::ImageView> image);
};

} // namespace iolink_cuda
