// ================================================================================ //
//       THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),        //
//                  AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                   //
//                                                                                  //
//     REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS      //
//     SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR      //
//                       WRITTEN AUTHORIZATION OF FEI S.A.S.                        //
//                                                                                  //
//                             RESTRICTED RIGHTS LEGEND                             //
//     USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS     //
//     WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN     //
//     SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT     //
//     CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN     //
//        TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.         //
//                                                                                  //
//                         COPYRIGHT (C) 2021 BY FEI S.A.S,                         //
//                                 BORDEAUX, FRANCE                                 //
//                               ALL RIGHTS RESERVED                                //
//                                                                                  //
//        SEE https://developer.openinventor.com/MiscFiles/EULA.pdf FOR MORE        //
// ================================================================================ //

#pragma once

#include <iolink/IOLinkAPI.h>
#include <iolink/Vector.h>

namespace iolink
{

/**
 * Describes a spatial axis system, with direction vectors describing the three axes.
 *
 * Axes are defined in a right hand system, so the third direction can be computed
 * from the first two ones.
 */
class IOLINK_API SpatialDirections final
{
public:
  /**
   * Create a default spatial direction.
   *
   * default values are the following:
   *  - column direction: (1.0, 0.0, 0.0)
   *  - row direction: (0.0, 1.0, 0.0)
   *  - slice direction: (0.0, 0.0, 1.0)
   */
  SpatialDirections();

  /**
   * Create a SpatialDirections instance from two vectors.
   *
   * The slice direction vector will be computed by using the cross
   * product of the two given direction.
   *
   * @param columnDirection column direction vector
   * @param rowDirection row direction vector
   */
  SpatialDirections(const Vector3d& columnDirection, const Vector3d& rowDirection);

  SpatialDirections(const SpatialDirections& other) = default;
  SpatialDirections& operator=(const SpatialDirections& other) = default;

  SpatialDirections(SpatialDirections&& other) noexcept = default;
  SpatialDirections& operator=(SpatialDirections&& other) noexcept = default;

  bool operator==(const SpatialDirections& other) const;
  bool operator!=(const SpatialDirections& other) const;

  /**
   * The direction vector following the column dimension (X)
   */
  inline const Vector3d& columnDirection() const { return m_columnDir; }

  /**
   * The direction vector following the row dimension (Y)
   */
  inline const Vector3d& rowDirection() const { return m_rowDir; }

  /**
   * The direction vector following the slice dimension (Z)
   */
  inline const Vector3d& sliceDirection() const { return m_sliceDir; }

  /**
   * Return a string decription of the spatial directions.
   */
  std::string toString() const;

private:
  Vector3d m_columnDir;
  Vector3d m_rowDir;
  Vector3d m_sliceDir;
};

} // end namespace iolink