// ================================================================================ //
//       THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),        //
//                  AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                   //
//                                                                                  //
//     REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS      //
//     SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR      //
//                       WRITTEN AUTHORIZATION OF FEI S.A.S.                        //
//                                                                                  //
//                             RESTRICTED RIGHTS LEGEND                             //
//     USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS     //
//     WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN     //
//     SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT     //
//     CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN     //
//        TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.         //
//                                                                                  //
//                         COPYRIGHT (C) 2021 BY FEI S.A.S,                         //
//                                 BORDEAUX, FRANCE                                 //
//                               ALL RIGHTS RESERVED                                //
//                                                                                  //
//        SEE https://developer.openinventor.com/MiscFiles/EULA.pdf FOR MORE        //
// ================================================================================ //

#pragma once

#include <iolink/FlagSet.h>
#include <iolink/IOLinkAPI.h>
#include <iolink/storage/DataAccess.h>

#include <cstdint>

namespace iolink
{

/**
 * Capabilities of a RandomAccess
 */
enum class RandomAccessCapability
{
  READ = 0x1,          ///< Data can be read
  WRITE = 0x2,         ///< Data can be written
  RESIZE = 0x4,        ///< Data can be resized
  MEMORY_ACCESS = 0x8, ///< Accessing internal CPU memory buffer

  READ_WRITE = 0x3,   ///< combination of READ and WRITE
  WRITE_RESIZE = 0x6, ///< combination of WRITE and RESIZE
};

extern template class IOLINK_API_IMPORT FlagSet<RandomAccessCapability>;

/** FlagSet for RandomAccessCapability enum */
using RandomAccessCapabilitySet = FlagSet<RandomAccessCapability>;

// Define bitwise operators for RandomAccessCapability
IOLINK_DEFINE_ENUM_BITWISE_OPERATORS(RandomAccessCapability)

/**
 * Interface representing a generic array-like accessor.
 *
 * This interface capabilities are indicated by the RandomAccessCapabilitySet
 * returned by the method RandomAccess::capabilities. Each capability enables
 * the use of a specific set of methods. When one of those methods is called,
 * if its corresponding capability is not supported, a NotImplemented exception
 * will be thrown.
 *
 *  @see RandomAccessCapability for capabilities managed by this object
 */
class IOLINK_API RandomAccess : public DataAccess
{
public:
  virtual ~RandomAccess() = default;

  /**
   * @return capabilities for current RandomAccess
   */
  virtual RandomAccessCapabilitySet capabilities() const = 0;

  /**
   * Checks if the RandomAccess supports the given capabilities.
   */
  inline bool support(RandomAccessCapabilitySet flags) { return this->capabilities().has(flags); }

  /**
   * Return the number of bytes of the resource accessed.
   */
  virtual size_t size() const = 0;

  /**
   * Return a string representation
   */
  std::string toString() const;

  // ==== //
  // READ //
  // ==== //

  /**
   * Read content of data into given buffer.
   *
   * Available when the RandomAccess has the READ capability.
   *
   * @param offset Offset in byte to the begining of read area
   * @param size Size in byte of read area
   * @param dst Destination buffer. This buffer should have a size of "size"
   *
   * @throw exception if offset is out of range
   *
   * @return Count of actually read bytes (can be zero or less than given size).
   */
  virtual size_t read(size_t offset, size_t size, void* dst);

  // ===== //
  // WRITE //
  // ===== //

  /**
   * Write content of src buffer into data.
   *
   * Available when the RandomAccess has the WRITE capability.
   *
   * @param offset Offset in byte to the beginning of write area
   * @param size Size in byte of write area
   * @param src Source buffer. This buffer should have a size of "size"
   *
   * @throw exception if offset is out of range
   *
   * @return Count of actually written bytes (can be zero or less than given size).
   */
  virtual size_t write(size_t offset, size_t size, const void* src);

  // ====== //
  // RESIZE //
  // ====== //

  /**
   * Resize content data.
   *
   * Available when the RandomAccess has the RESIZE capability.
   *
   * @param newSize New size of data. Content of data is undefined after being resized.
   */
  virtual void resize(size_t newSize);

  // ============= //
  // MEMORY_ACCESS //
  // ============= //

  /**
   * Return a mutable pointer to the internal CPU memory buffer.
   *
   * Available when the RandomAccess has the MEMORY_ACCESS capability.
   *
   * @throw NotImplemented If the implementation does not support memory access.
   */
  virtual void* buffer();

  /**
   * Return an immutable pointer to the internal CPU memory buffer.
   *
   * Available when the RandomAccess has the MEMORY_ACCESS capability.
   *
   * @throw NotImplemented If the implementation does not support memory access.
   */
  virtual const void* bufferReadOnly() const;

  /**
   * Size in bytes of the internal CPU buffer.
   *
   * Available when the RandomAccess has the MEMORY_ACCESS capability.
   *
   * @throw NotImplemented If the implementation does not support memory access.
   */
  virtual size_t bufferSize();
};

} // namespace iolink
