// ================================================================================ //
//       THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),        //
//                  AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                   //
//                                                                                  //
//     REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS      //
//     SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR      //
//                       WRITTEN AUTHORIZATION OF FEI S.A.S.                        //
//                                                                                  //
//                             RESTRICTED RIGHTS LEGEND                             //
//     USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS     //
//     WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN     //
//     SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT     //
//     CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN     //
//        TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.         //
//                                                                                  //
//                         COPYRIGHT (C) 2021 BY FEI S.A.S,                         //
//                                 BORDEAUX, FRANCE                                 //
//                               ALL RIGHTS RESERVED                                //
//                                                                                  //
//        SEE https://developer.openinventor.com/MiscFiles/EULA.pdf FOR MORE        //
// ================================================================================ //

#pragma once

#include <cstdint>

#include <iolink/FlagSet.h>
#include <iolink/IOLinkAPI.h>
#include <iolink/storage/DataAccess.h>

namespace iolink
{

/**
 * Capabilities of a StreamAccess
 */
enum class StreamAccessCapability
{
  READ = 0x1,          ///<  Data can be read
  WRITE = 0x2,         ///<  Data can be written
  SEEK = 0x4,          ///<  Cursor for read/write can be moved
  MEMORY_ACCESS = 0x8, ///< Accessing internal CPU memory buffer

  READ_WRITE = 0x3, ///<  combination of READ and WRITE
  READ_SEEK = 0x5,  ///<  combination of READ and SEEK
  WRITE_SEEK = 0x6, ///<  combination of WRITE and SEEK
};

extern template class IOLINK_API_IMPORT FlagSet<StreamAccessCapability>;

/** FlagSet for StreamAccessCapability enum */
using StreamAccessCapabilitySet = FlagSet<StreamAccessCapability>;

// Define bitwise operators for RandomAccessCapability
IOLINK_DEFINE_ENUM_BITWISE_OPERATORS(StreamAccessCapability)

/**
 * Position indicator for seek method
 */
enum class SeekOrigin
{
  /** Begining of the stream */
  BEGIN = 1,
  /** Current position of the stream */
  CURRENT = 2,
  /** End of the stream */
  END = 3,
};

/**
 * Mother interface for all stream-like data accessors.
 *
 * This interface capabilities are indicated by the StreamAccessCapabilitySet
 * returned by the method StreamAccess::capabilities. Each capability enables
 * the use of a specific set of methods. When one of those methods is called,
 * if its corresponding capability is not supported, a NotImplemented exception
 * will be thrown.
 *
 *  @see StreamAccessCapability for capabilities managed by this object
 */
class IOLINK_API StreamAccess : public DataAccess
{
public:
  virtual ~StreamAccess() = default;

  /**
   * @return capabilities for current stream
   */
  virtual StreamAccessCapabilitySet capabilities() const = 0;

  /**
   * Checks if the StreamAccess supports the given capabilities.
   */
  inline bool support(StreamAccessCapabilitySet flags) { return this->capabilities().has(flags); }

  /**
   * Checks if the stream is currently valid for reading (only)
   * Default implementation returns true if nor 'end of file'
   * or 'fail' status are raised
   * @return true if the stream can be read
   */
  virtual bool good() const;

  /**
   * Checks if the cursor is at the end of the stream for reading
   * For writing, this status is not relevant.
   *
   * @return true if the end of stream is reached. No more byte can be read.
   */
  virtual bool eof() const = 0;

  /*
   * Checks if an error occured on the stream which forbids its use
   * Remark: 'end of file' is not considered as an error
   *
   * Default implementation returns false.
   *
   * @return true if an error occured, false otherwise
   */
  virtual bool fail() const = 0;

  /**
   * Return a string representation
   */
  std::string toString() const;

  // ==== //
  // READ //
  // ==== //

  /**
   * Read data from cursor current position into destination buffer.
   * Cursor position is incremented by the actually read bytes count.
   *
   * Available when the StreamAccess has the READ capability.
   *
   * @param size Size in byte of the area to read
   * @param dst Destination buffer. This buffer should have a size of "size"
   *
   * @return Count of actually read bytes.
   */
  virtual size_t read(size_t size, void* dst);

  /**
   * Return the current byte without moving the cursor.
   *
   * Available when the StreamAccess has the READ capability.
   *
   * @throw Error if cursor is at EOF.
   */
  virtual uint8_t peek();

  // ===== //
  // WRITE //
  // ===== //

  /**
   * Write content of src buffer into StreamAccess.
   * Stream cursor is incremented by actually written bytes count.
   *
   * Available when the StreamAccess has the WRITE capability.
   *
   * @param size Size in byte of given buffer to write
   * @param src Source buffer. This buffer should have a size of "size"
   *
   * @return Count of actually written bytes. It may not be considered as an error
   * that return value if different than size (ie. an exception may not be thrown).
   * It is up to user to decide what to do in this case.
   */
  virtual size_t write(size_t size, const void* src);

  /**
   * Assure that all data is written to the endpoint.
   *
   * Available when the StreamAccess has the WRITE capability.
   */
  virtual void flush();

  // ==== //
  // SEEK //
  // ==== //

  /**
   * Returns the current position of the cursor.
   *
   * Available when the StreamAccess has the SEEK capability.
   */
  virtual size_t tell();

  /**
   * Move the cursor to the desired position.
   *
   * Available when the StreamAccess has the SEEK capability.
   *
   * Cursor position must be set as following:
   *    new position = current position + offset
   * If new computed position is less than 0, previous current position is kept.
   *
   * @param offset Offset used to move the cursor according to the origin.
   * @param origin The origin from which to move the cursor. Can be the start
   * of the stream, the current position of the cursor, or the end of the stream.
   * Positive offsets always make the cursor move forward in the stream, whatever the origin.
   * Negative offsets always make the cursor move backward.
   *
   * @throw Error if trying to set cursor before BEGIN position (negative value)
   */
  virtual void seek(int64_t offset, SeekOrigin origin);

  // ============= //
  // MEMORY_ACCESS //
  // ============= //

  /**
   * Return a mutable pointer to the internal CPU memory buffer.
   *
   * Available when the RandomAccess has the MEMORY_ACCESS capability.
   *
   * @throw NotImplemented If the implementation does not support memory access.
   */
  virtual void* buffer();

  /**
   * Return an immutable pointer to the internal CPU memory buffer.
   *
   * Available when the RandomAccess has the MEMORY_ACCESS capability.
   *
   * @throw NotImplemented If the implementation does not support memory access.
   */
  virtual const void* bufferReadOnly() const;

  /**
   * Size in bytes of the internal CPU buffer.
   *
   * Available when the RandomAccess has the MEMORY_ACCESS capability.
   *
   * @throw NotImplemented If the implementation does not support memory access.
   */
  virtual size_t bufferSize();
};

} // namespace iolink
