// ================================================================================ //
//       THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),        //
//                  AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                   //
//                                                                                  //
//     REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS      //
//     SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR      //
//                       WRITTEN AUTHORIZATION OF FEI S.A.S.                        //
//                                                                                  //
//                             RESTRICTED RIGHTS LEGEND                             //
//     USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS     //
//     WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN     //
//     SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT     //
//     CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN     //
//        TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.         //
//                                                                                  //
//                         COPYRIGHT (C) 2021 BY FEI S.A.S,                         //
//                                 BORDEAUX, FRANCE                                 //
//                               ALL RIGHTS RESERVED                                //
//                                                                                  //
//        SEE https://developer.openinventor.com/MiscFiles/EULA.pdf FOR MORE        //
// ================================================================================ //

#pragma once

#include <memory>

#include <iolink/DataType.h>
#include <iolink/FlagSet.h>
#include <iolink/IOLinkAPI.h>
#include <iolink/Indexer.h>
#include <iolink/RegionX.h>
#include <iolink/VectorX.h>
#include <iolink/metadata/MetadataNode.h>
#include <iolink/property/ImageProperties.h>
#include <iolink/view/ParallelAccessExtension.h>
#include <iolink/view/TilingInfoExtension.h>
#include <iolink/view/View.h>

namespace iolink
{

/**
 * Define capabilities of an ImageView.
 */
enum class ImageCapability
{
  // Flags
  READ = 0x1,          ///< Data and metadata read capability
  WRITE = 0x2,         ///< Data, metadata, and properties write capability
  RESHAPE = 0x4,       ///< Reshaping capabilities
  MEMORY_ACCESS = 0x8, ///< Accessing internal CPU memory buffer

  // Combinations
  READ_WRITE = 0x3, ///< Combines READ and WRITE
};

extern template class IOLINK_API_IMPORT FlagSet<ImageCapability>;

/**
 * FlagSet to handle ImageView capabilities
 */
using ImageCapabilitySet = FlagSet<ImageCapability>;

// Define bitwise operators for ImageCapability
IOLINK_DEFINE_ENUM_BITWISE_OPERATORS(ImageCapability)

/**
 * Interface representing an N dimensional image
 *
 * This class can handle multidimensional image, for example:
 * - A 2D image has 2 dimensions.
 * - A 3D Volume has 3 dimensions.
 * - A 2D time series has 3 dimensions, where 3rd one represent time.
 *
 * This interface capabilities are indicated by the ImageCapabilitySet
 * returned by the method ImageView::capabilities. Each capability enable
 * the use of a specific set of methods. When one of those methods is
 * called, if its corresponding capability is not supported, a NotImplemented
 * exception will be thrown.
 *
 * An ImageView is composed of the following parts:
 *  - A shape, that describes the size of the dataset along each of its axes.
 *  - A data type, giving info on what is stored on each element of the dataset.
 *  - Properties, giving generic info on the image, such as its spatial calibration,
 *    the semantic of its dimensions, and detailed information on its elements.
 *  - Metadata, hierarchical tree of key/value pair, describing arbitrary information.
 *  - Image's data itself.
 *
 * @see ImageCapability
 */
class IOLINK_API ImageView : public View
{

public:
  virtual ~ImageView() = default;

  /**
   * Returns the dimension of the View.
   */
  inline size_t dimensionCount() { return this->shape().size(); }

  /**
   * Checks if the ImageView supports the given capabilities.
   */
  inline bool support(ImageCapabilitySet flags) { return this->capabilities().has(flags); }

  /**
   * Return the capabilities of this image.
   */
  virtual ImageCapabilitySet capabilities() const = 0;

  /**
   * Return the shape of the image, each component containing the size in this dimension.
   *
   * Example:
   *  - A 2D image will return a vector with two dimension, the first being the number
   *    of columns, and the second being the number of rows.
   *
   *  - A 3D image will return a vector with two dimension, the first being the number
   *    of columns, the second being the number of rows, and the third being the number
   *    of slices.
   */
  virtual const VectorXu64& shape() = 0;

  /**
   * Return the data type of the samples of this image.
   */
  virtual DataType dataType() = 0;

  /**
   * Return the view's properties.
   */
  virtual std::shared_ptr<const ImageProperties> properties() = 0;

  /**
   * Return a string representation
   */
  std::string toString() const;

  // ==== //
  // READ //
  // ==== //

  /**
   * Return the root node of the metadata attached to this image.
   *
   * Available when the ImageView has the READ capability.
   *
   * @throw NotImplemented If the implementation does not support reading.
   */
  virtual std::shared_ptr<const MetadataNode> metadata() const;

  /**
   * Read the data at the given index into a buffer.
   *
   * Available when the ImageView has the READ capability.
   *
   * Default implementation will call the readRegion method
   * with a region of one pixel.
   *
   * @param index the index of the sample to read.
   * @param dst The buffer where read data will be copied.
   */
  virtual void read(const VectorXu64& index, void* dst);

  /**
   * Read a region of the image into a buffer.
   *
   * Available when the ImageView has the READ capability.
   *
   * @param region The region of the image to read from.
   * @param dst The buffer where read data will be copied.
   *
   * @throw NotImplemented If the implementation does not support reading.
   */
  virtual void readRegion(const RegionXu64& region, void* dst);

  // Shortcuts

  /**
   * Shortcut to access Image interpretation from image properties
   *
   * @return interpretation of the image
   */
  ImageInterpretation imageInterpretation();

  /**
   * Shortcut to access axes interpretation from image properties
   *
   * @return axes interpretation of the image
   */
  ImageType axesInterpretation();

  /**
   * Shortcut to access spatial origin from image properties
   *
   * @return spatial origin of the image
   */
  Vector3d spatialOrigin();

  /**
   * Shortcut to access spatial spacing from image properties
   *
   * @return spatial spacing of the image
   */
  Vector3d spatialSpacing();

  /**
   * Shortcut to access the spatial unit from image properties
   *
   * @return spatial unit of the image
   */
  std::string spatialUnit();

  /**
   * Shortcut to access spatial directions from image properties
   *
   * @return spatial directions of the image
   */
  const SpatialDirections& spatialDirections();

  /**
   * Shortcut to access the sample bitdepth from image properties
   *
   * @return bitdepth of the image
   */
  size_t bitDepth();

  /**
   * Shortcut to access hasAlpha status from image properties
   *
   * @return TRUE if image has an alpha channel, FALSE otherwise
   */
  bool hasAlpha();

  /**
   * Shortcut to access the sample value range from image properties
   *
   * @return value range for the image samples
   */
  Vector2d valueRange();

  // ===== //
  // WRITE //
  // ===== //

  /**
   * Update the image's properties.
   *
   * Available when the ImageView has the WRITE capability.
   *
   * @throw NotImplemented If the implementation does not support writing.
   */
  virtual void setProperties(std::shared_ptr<const ImageProperties> properties);

  /**
   * Attach a new metadata node to the image.
   *
   * Available when the ImageView has the WRITE capability.
   *
   * @throw NotImplemented If the implementation does not support writing.
   */
  virtual void setMetadata(std::shared_ptr<MetadataNode> metadata);

  /**
   * Write value from a buffer at the given index.
   *
   * Available when the ImageView has the WRITE capability.
   *
   * Default implementation will call the writeRegion method
   * with a region of one pixel.
   *
   * @param index The index of the image's sample to write.
   * @param src The buffer containing the data that will be written.
   */
  virtual void write(const VectorXu64& index, const void* src);

  /**
   * Write a input buffer into a given region of the image
   *
   * Available when the ImageView has the WRITE capability.
   *
   * @param region The region to write.
   * @param src The buffer containing the data that will be written.
   *
   * @throw NotImplemented If the implementation does not support writing.
   */
  virtual void writeRegion(const RegionXu64& region, const void* src);

  // Shortcuts

  /**
   * Shortcut to set Image interpretation into image properties
   *
   * @param interpretation Image interpretation to set
   */
  void setImageInterpretation(const ImageInterpretation interpretation);

  /**
   * Shortcut to set axes interpretation into image properties
   *
   * @param type Axes interpretation to set
   */
  void setAxesInterpretation(const ImageType type);

  /**
   * Shortcut to set spatial origin into image properties
   *
   * @param origin spatial origin to set
   */
  void setSpatialOrigin(const Vector3d& origin);

  /**
   * Shortcut to set spatial spacing into image properties
   *
   * @param spacing spatial spacing to set
   */
  void setSpatialSpacing(const Vector3d& spacing);

  /**
   * Shortcut to set spatial unit into image properties
   *
   * @param unit spatial unit to set
   */
  void setSpatialUnit(const std::string& unit);

  /**
   * Shortcut to set spatial direcitons into image properties
   *
   * @param directions spatial directions to set
   */
  void setSpatialDirections(const SpatialDirections& directions);

  /**
   * Shortcut to set sample bitdepth into image properties
   *
   * @param bitDepth bitDepth to set
   */
  void setBitDepth(size_t bitDepth);

  /**
   * Shortcut to indicate the presence of an alpha channel into image properties
   *
   * @param value TRUE if image has an alpha channel, FALSE otherwise
   */
  void setAlpha(bool value);

  /**
   * Shortcut to set sample value range into image properties
   *
   * @param range value range to set
   */
  void setValueRange(const Vector2d& range);

  // ======= //
  // RESHAPE //
  // ======= //

  /**
   * Change the shape and sample data type of the image.
   *
   * Available when the ImageView has the RESHAPE capability.
   *
   * All the image's data is invalidated after this operation.
   *
   * @param shape The new shape.
   * @param dataType The new sample data type.
   *
   * @throw NotImplemented If the implementation does not support reshaping.
   */
  virtual void reshape(const VectorXu64& shape, DataType dataType);

  // ============= //
  // MEMORY_ACCESS //
  // ============= //

  /**
   * Return an Indexer describing the memory layout of the buffer.
   *
   * Available when the ImageView has the MEMORY_ACCESS capability.
   *
   * @throw NotImplemented If the implementation does not support memory access.
   */
  virtual const Indexer& indexer();

  /**
   * Return a mutable pointer to the internal CPU memory buffer.
   *
   * Available when the ImageView has the MEMORY_ACCESS capability.
   *
   * @throw NotImplemented If the implementation does not support memory access.
   */
  virtual void* buffer();

  /**
   * Return an immutable pointer to the internal CPU memory buffer.
   *
   * Available when the ImageView has the MEMORY_ACCESS capability.
   *
   * @throw NotImplemented If the implementation does not support memory access.
   */
  virtual const void* bufferReadOnly() const;

  /**
   * Size in bytes of the internal CPU buffer.
   *
   * Available when the ImageView has the MEMORY_ACCESS capability.
   *
   * @throw NotImplemented If the implementation does not support memory access.
   */
  virtual size_t bufferSize();

  // ========== //
  // Extensions //
  // ========== //

  /**
   * Shortcut to retrieve TilingInfo extension.
   */
  std::shared_ptr<TilingInfoExtension> tilingInfo();

  /**
   * Shortcut to retrieve ParallelAccess extension
   */
  std::shared_ptr<ParallelAccessExtension> parallelAccess();
};

} // namespace iolink
