/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Data/MeasurementInformation/FieldInformation.h>
#include <ImageDev/ImageDevCppExports.h>
#include <iolink/Vector.h>
#include <memory>

namespace object_model {
class DynamicDictionary;
class Array;
} // namespace object_model

namespace imagedev {

namespace measurements {

/// Data type of a scalar value.
enum DataType
{
    /// Unsigned 64-bit integer value.
    UNSIGNED_INTEGER = 0,
    /// Signed 64-bit integer value.
    SIGNED_INTEGER,
    /// 64-bit float value
    FLOATING
};

struct MeasurementPrivate;

/// Base class of all measurements that can be computed by any analysis algorithm with user-defined measurements.
class IMAGEDEV_CPP_API Measurement
{
public:
    typedef std::shared_ptr< Measurement > Ptr;

    /// Constructor
    Measurement( object_model::DynamicDictionary* object );

    /// Gets the name of the measurement.
    virtual std::string name() const = 0;
    /// Gets the physical unit of the measurement.
    virtual std::string physDim() const = 0;
    /// Initializes the Array if needed
    void initArray();

    /// Returns the value at the index position in the analysis and converts it to a 64-bit float.
    double toDouble( const iolink::VectorX< uint64_t >& index ) const;

    /// Returns the value at the index position in the analysis and converts it to an unsigned 64-bit integer.
    /// Caution:
    /// - If the original value is floating point, a precision loss will occur.
    /// - If the original value is negative, the returned value will be inconsistent.
    uint64_t toUInt64( const iolink::VectorX< uint64_t >& index ) const;

    /// Returns the value at the index position in the analysis and converts it to a signed 64-bit integer.
    /// Caution:
    /// - If the original value is floating point, a precision loss will occur.
    int64_t toInt64( const iolink::VectorX< uint64_t >& index ) const;

    /// Returns the shape of the array.
    iolink::VectorX< uint64_t > shape() const;

    /// Returns the data type of the measurement.
    DataType dataType() const;
    // return information about the measurement
    const FieldInformation& information() const;

    /// Destructor
    virtual ~Measurement();

protected:
    /// init m_array if needed and return it
    object_model::Array* getArray();
    const object_model::Array* getArray() const;

private:
    friend struct MeasurementPrivate;
    // private impl
    MeasurementPrivate* m_p;
};

} // namespace measurements
} // namespace imagedev
