/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Data/Kernels/MorphologicalKernel2d.h>
#include <ImageDev/ImageDevCppExports.h>
#include <memory>

namespace imagedev {

class MorphologicalLut2dPrivate;

/**
 * This class represents a 2D morphological lut.
 */
class IMAGEDEV_CPP_API MorphologicalLut2d
{
public:
    typedef MorphologicalLut2d Self;
    typedef std::shared_ptr< Self > Ptr;

    // constructor returning a shared ptr on an empty MorphologicalLut2d
    static Ptr New();

    // constructor returning a shared ptr on a MorphologicalLut2d with a kernel with a specific rotation
    static Ptr New( const MorphologicalKernel2d::Ptr kernel, const size_t rotation );

    // constructor returning a shared ptr on a MorphologicalLut2d with a kernel in all possible rotations
    static Ptr New( const MorphologicalKernel2d::Ptr kernel );

    // destructor
    ~MorphologicalLut2d();

    /**
     * Add a kernel to the LUT
     * @param[in] kernel The kernel to add into the LUT.
     * A 2D-kernel should contain exactly 9 values and indexes matching the following table.
     * 4 3 2
     * 5 0 1
     * 6 7 8
     * Accepted values for the kernel are MorphologicalKernel2d::Value:
     * - ZERO: pixel is off
     * - ONE: pixel is on
     * - ANY: pixel is ignored
     * @param[in] rotation number of rotation of 2 PI / 8 of the kernel before insertion.
     */
    void addKernel( const MorphologicalKernel2d::Ptr kernel, const size_t rotation = 0 );

    /**
     * Add a kernel to the LUT with all possible rotations
     * @param[in] kernel The kernel to add into the LUT.
     * A 2D-kernel should contain exactly 9 values and indexes matching the following table.
     * 4 3 2
     * 5 0 1
     * 6 7 8
     * Accepted values for the kernel are MorphologicalKernel2d::Value:
     * - ZERO: pixel is off
     * - ONE: pixel is on
     * - ANY: pixel is ignored
     */
    void addKernels( const MorphologicalKernel2d::Ptr kernel );

    /**
     * Reset the LUT to initial values (empty).
     */
    void reset();

    /**
     * Return the number of kernels in the lut.
     */
    size_t kernelCount();

    friend class MorphologicalLut2dPrivate;

protected:
    MorphologicalLut2d();
    MorphologicalLut2dPrivate* m_p;
};

} // namespace imagedev
