/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <iolink/Vector.h>
#include <iolink/view/ImageView.h>
#include <ImageDev/Data/GlobalMeasurements/AutoThresholdingMsr.h>

namespace imagedev
{
/// Computes a threshold value partitioning automatically a gray level image into two classes.
class IMAGEDEV_CPP_API AutoThresholdingValue final : public GenericAlgorithm
{
public:
    /// The way to determine the input intensity range.
    enum RangeMode
    {
    /// The histogram is computed between the minimum and the maximum of the image.
        MIN_MAX = 0,
    /// The histogram is computed between user-defined bounds [a,b].
        OTHER
    };
    /// The criterion to compute the threshold from the histogram.
    enum ThresholdCriterion
    {
    /// The measurement of dispersion used in the algorithm is the entropy of the intensity distribution.
        ENTROPY = 0,
    /// The measurement of dispersion used in the algorithm is the variance of the intensity distribution (also known as the Otsu method).
        FACTORISATION,
    /// The measurement of dispersion used in the algorithm is the moments of the intensity distribution.
        MOMENTS,
    /// The measurement of dispersion used in the algorithm is the isodata of the intensity distribution.
        ISODATA
    };

    // Command constructor.
    AutoThresholdingValue();


    /// Gets the inputGrayImage parameter.
    /// The input grayscale image
    std::shared_ptr< iolink::ImageView > inputGrayImage() const;
    /// Sets the inputGrayImage parameter.
    /// The input grayscale image
    void setInputGrayImage( std::shared_ptr< iolink::ImageView > inputGrayImage );

    /// Gets the rangeMode parameter.
    /// The way to determine the input intensity range.
    AutoThresholdingValue::RangeMode rangeMode() const;
    /// Sets the rangeMode parameter.
    /// The way to determine the input intensity range.
    void setRangeMode( const AutoThresholdingValue::RangeMode& rangeMode );

    /// Gets the intensityInputRange parameter.
    /// The input intensity range [a,b] inside which the threshold is searched. This parameter is ignored if the range mode is set to MIN_MAX.
    iolink::Vector2d intensityInputRange() const;
    /// Sets the intensityInputRange parameter.
    /// The input intensity range [a,b] inside which the threshold is searched. This parameter is ignored if the range mode is set to MIN_MAX.
    void setIntensityInputRange( const iolink::Vector2d& intensityInputRange );

    /// Gets the thresholdCriterion parameter.
    /// The criterion to compute the threshold from the histogram.
    AutoThresholdingValue::ThresholdCriterion thresholdCriterion() const;
    /// Sets the thresholdCriterion parameter.
    /// The criterion to compute the threshold from the histogram.
    void setThresholdCriterion( const AutoThresholdingValue::ThresholdCriterion& thresholdCriterion );

    /// Gets the outputMeasurement parameter.
    /// The output measurement containing the computed threshold.
    AutoThresholdingMsr::Ptr outputMeasurement() const;

    // Method to launch the command.
    void execute();

};

/// Computes a threshold value partitioning automatically a gray level image into two classes.
/// @param inputGrayImage The input grayscale image
/// @param rangeMode The way to determine the input intensity range.
/// @param intensityInputRange The input intensity range [a,b] inside which the threshold is searched. This parameter is ignored if the range mode is set to MIN_MAX.
/// @param thresholdCriterion The criterion to compute the threshold from the histogram.
/// @param outputMeasurement The output measurement containing the computed threshold.
/// @return Returns the outputMeasurement output parameter.
IMAGEDEV_CPP_API 
AutoThresholdingMsr::Ptr
autoThresholdingValue( std::shared_ptr< iolink::ImageView > inputGrayImage,
                       AutoThresholdingValue::RangeMode rangeMode,
                       const iolink::Vector2d& intensityInputRange,
                       AutoThresholdingValue::ThresholdCriterion thresholdCriterion,
                       AutoThresholdingMsr::Ptr outputMeasurement = nullptr );
} // namespace imagedev
