/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <iolink/view/ImageView.h>
#include <ImageDev/Data/GlobalMeasurements/CorrelationMsr.h>

namespace imagedev
{
// Output structure of the binaryCorrelation2d function.
struct BinaryCorrelation2dOutput
{
    /// The output correlation image. Its dimensions are forced to the same values as the input. Its data type is forced to floating point.
    std::shared_ptr< iolink::ImageView > outputImage;
    /// The correlation matching results.
    CorrelationMsr::Ptr outputMeasurement;
};

/// Performs a logical correlation between a binary two-dimensional image and a binary kernel.
class IMAGEDEV_CPP_API BinaryCorrelation2d final : public GenericAlgorithm
{
public:
    /// The calculation offset, in pixels. The greater this value, computation is faster but detection is less precise.
    enum OffsetMode
    {
    /// The correlation is computed with a step of 1.
        OFFSET_1 = 0,
    /// The correlation is computed with a step of 2.
        OFFSET_2,
    /// The correlation is computed with a step of 4.
        OFFSET_4,
    /// The correlation is computed with a step of 8.
        OFFSET_8
    };

    // Command constructor.
    BinaryCorrelation2d();


    /// Gets the inputBinaryImage parameter.
    /// The input binary image.
    std::shared_ptr< iolink::ImageView > inputBinaryImage() const;
    /// Sets the inputBinaryImage parameter.
    /// The input binary image.
    void setInputBinaryImage( std::shared_ptr< iolink::ImageView > inputBinaryImage );

    /// Gets the inputKernelImage parameter.
    /// The correlation kernel.
    std::shared_ptr< iolink::ImageView > inputKernelImage() const;
    /// Sets the inputKernelImage parameter.
    /// The correlation kernel.
    void setInputKernelImage( std::shared_ptr< iolink::ImageView > inputKernelImage );

    /// Gets the offsetMode parameter.
    /// The calculation offset, in pixels. The greater this value, computation is faster but detection is less precise.
    BinaryCorrelation2d::OffsetMode offsetMode() const;
    /// Sets the offsetMode parameter.
    /// The calculation offset, in pixels. The greater this value, computation is faster but detection is less precise.
    void setOffsetMode( const BinaryCorrelation2d::OffsetMode& offsetMode );

    /// Gets the outputImage parameter.
    /// The output correlation image. Its dimensions are forced to the same values as the input. Its data type is forced to floating point.
    std::shared_ptr< iolink::ImageView > outputImage() const;
    /// Sets the outputImage parameter.
    /// The output correlation image. Its dimensions are forced to the same values as the input. Its data type is forced to floating point.
    void setOutputImage( std::shared_ptr< iolink::ImageView > outputImage );

    /// Gets the outputMeasurement parameter.
    /// The correlation matching results.
    CorrelationMsr::Ptr outputMeasurement() const;

    // Method to launch the command.
    void execute();

};

/// Performs a logical correlation between a binary two-dimensional image and a binary kernel.
/// @param inputBinaryImage The input binary image.
/// @param inputKernelImage The correlation kernel.
/// @param offsetMode The calculation offset, in pixels. The greater this value, computation is faster but detection is less precise.
/// @param outputImage The output correlation image. Its dimensions are forced to the same values as the input. Its data type is forced to floating point.
/// @param outputMeasurement The correlation matching results.
/// @return Returns a BinaryCorrelation2dOutput structure containing the outputImage and outputMeasurement output parameters.
IMAGEDEV_CPP_API 
BinaryCorrelation2dOutput
binaryCorrelation2d( std::shared_ptr< iolink::ImageView > inputBinaryImage,
                     std::shared_ptr< iolink::ImageView > inputKernelImage,
                     BinaryCorrelation2d::OffsetMode offsetMode,
                     std::shared_ptr< iolink::ImageView > outputImage = nullptr,
                     CorrelationMsr::Ptr outputMeasurement = nullptr );
} // namespace imagedev
