/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <iolink/Vector.h>
#include <ImageDev/Data/Cuda/CudaContext.h>
#include <iolink/view/ImageView.h>

namespace imagedev
{
/// Applies a three-dimensional Gaussian filter using either a separable finite kernel.
/// Warning: This command is experimental, his signature may be modified between now and his final version.
class IMAGEDEV_CPP_API CudaGaussianFilter3d final : public GenericAlgorithm
{
public:
    /// The algorithm implementation used to compute the gaussian filter.
    enum FilterMode
    {
    /// This mode uses a Finite Impulse Response algorithm on the X, Y, and Z directions.
        SEPARABLE = 0,
    };
    /// The output data type. It can either be the same as the input type, or forced to be float. In the case of floating input images, this parameter has no effect.
    enum OutputType
    {
    /// The output image has same type as the input image. In the case of integer images, this mode can lead to a loss of precision.
        SAME_AS_INPUT = 0,
    /// The output image type is forced to floating point.
        FLOAT_32_BIT
    };
    /// The way to manage the GPU memory.
    enum TilingMode
    {
    /// The entire input image is transferred to the GPU memory.
        NONE = 0,
    /// The input image is processed by tiles of a predefined size.
        USER_DEFINED
    };

    // Command constructor.
    CudaGaussianFilter3d();


    /// Gets the inputImage parameter.
    /// The input image. The type of image can be integer or float.@BR
    std::shared_ptr< iolink::ImageView > inputImage() const;
    /// Sets the inputImage parameter.
    /// The input image. The type of image can be integer or float.@BR
    void setInputImage( std::shared_ptr< iolink::ImageView > inputImage );

    /// Gets the filterMode parameter.
    /// The algorithm implementation used to compute the gaussian filter.
    CudaGaussianFilter3d::FilterMode filterMode() const;
    /// Sets the filterMode parameter.
    /// The algorithm implementation used to compute the gaussian filter.
    void setFilterMode( const CudaGaussianFilter3d::FilterMode& filterMode );

    /// Gets the standardDeviation parameter.
    /// The standard deviation value for each direction (X, Y, Z) in voxel units. Each value must be greater or equal to 0.0.
    iolink::Vector3d standardDeviation() const;
    /// Sets the standardDeviation parameter.
    /// The standard deviation value for each direction (X, Y, Z) in voxel units. Each value must be greater or equal to 0.0.
    void setStandardDeviation( const iolink::Vector3d& standardDeviation );

    /// Gets the kernelSizeFactor parameter.
    /// This parameter is used to compute the size of the kernel applied in the SEPARABLE mode. The kernel size value is twice the kernelSizeFactor multiplied by the standard deviation associated with the axis. If the resulting kernel size is even, it is incremented by one in order to ensure an odd kernel size.
    double kernelSizeFactor() const;
    /// Sets the kernelSizeFactor parameter.
    /// This parameter is used to compute the size of the kernel applied in the SEPARABLE mode. The kernel size value is twice the kernelSizeFactor multiplied by the standard deviation associated with the axis. If the resulting kernel size is even, it is incremented by one in order to ensure an odd kernel size.
    void setKernelSizeFactor( const double& kernelSizeFactor );

    /// Gets the outputType parameter.
    /// The output data type. It can either be the same as the input type, or forced to be float. In the case of floating input images, this parameter has no effect.
    CudaGaussianFilter3d::OutputType outputType() const;
    /// Sets the outputType parameter.
    /// The output data type. It can either be the same as the input type, or forced to be float. In the case of floating input images, this parameter has no effect.
    void setOutputType( const CudaGaussianFilter3d::OutputType& outputType );

    /// Gets the lowMemory parameter.
    /// This parameter defines if the SEPARABLE algorithm must limit its memory usage. If equal to false, a temporary 32-bit float image is used to store the result before casting it to the output type. The result is thus less precise and faster to compute when this parameter is set to true.
    bool lowMemory() const;
    /// Sets the lowMemory parameter.
    /// This parameter defines if the SEPARABLE algorithm must limit its memory usage. If equal to false, a temporary 32-bit float image is used to store the result before casting it to the output type. The result is thus less precise and faster to compute when this parameter is set to true.
    void setLowMemory( const bool& lowMemory );

    /// Gets the tilingMode parameter.
    /// The way to manage the GPU memory.
    CudaGaussianFilter3d::TilingMode tilingMode() const;
    /// Sets the tilingMode parameter.
    /// The way to manage the GPU memory.
    void setTilingMode( const CudaGaussianFilter3d::TilingMode& tilingMode );

    /// Gets the tileSize parameter.
    /// The tile width and height in pixels. This parameter is used only in USER_DEFINED tiling mode.
    iolink::Vector3u32 tileSize() const;
    /// Sets the tileSize parameter.
    /// The tile width and height in pixels. This parameter is used only in USER_DEFINED tiling mode.
    void setTileSize( const iolink::Vector3u32& tileSize );

    /// Gets the cudaContext parameter.
    /// CUDA context information.
    CudaContext::Ptr cudaContext() const;
    /// Sets the cudaContext parameter.
    /// CUDA context information.
    void setCudaContext( const CudaContext::Ptr& cudaContext );

    /// Gets the outputImage parameter.
    /// The output image. Dimensions, calibration, and interpretation of the output image are forced to the same values as the input.
    std::shared_ptr< iolink::ImageView > outputImage() const;
    /// Sets the outputImage parameter.
    /// The output image. Dimensions, calibration, and interpretation of the output image are forced to the same values as the input.
    void setOutputImage( std::shared_ptr< iolink::ImageView > outputImage );

    // Method to launch the command.
    void execute();

};

/// Applies a three-dimensional Gaussian filter using either a separable finite kernel.
/// Warning: This command is experimental, his signature may be modified between now and his final version.
/// @param inputImage The input image. The type of image can be integer or float.@BR
/// @param filterMode The algorithm implementation used to compute the gaussian filter.
/// @param standardDeviation The standard deviation value for each direction (X, Y, Z) in voxel units. Each value must be greater or equal to 0.0.
/// @param kernelSizeFactor This parameter is used to compute the size of the kernel applied in the SEPARABLE mode. The kernel size value is twice the kernelSizeFactor multiplied by the standard deviation associated with the axis. If the resulting kernel size is even, it is incremented by one in order to ensure an odd kernel size.
/// @param outputType The output data type. It can either be the same as the input type, or forced to be float. In the case of floating input images, this parameter has no effect.
/// @param lowMemory This parameter defines if the SEPARABLE algorithm must limit its memory usage. If equal to false, a temporary 32-bit float image is used to store the result before casting it to the output type. The result is thus less precise and faster to compute when this parameter is set to true.
/// @param tilingMode The way to manage the GPU memory.
/// @param tileSize The tile width and height in pixels. This parameter is used only in USER_DEFINED tiling mode.
/// @param cudaContext CUDA context information.
/// @param outputImage The output image. Dimensions, calibration, and interpretation of the output image are forced to the same values as the input.
/// @return Returns the outputImage output parameter.
IMAGEDEV_CPP_API 
std::shared_ptr< iolink::ImageView >
cudaGaussianFilter3d( std::shared_ptr< iolink::ImageView > inputImage,
                      CudaGaussianFilter3d::FilterMode filterMode,
                      const iolink::Vector3d& standardDeviation,
                      double kernelSizeFactor,
                      CudaGaussianFilter3d::OutputType outputType,
                      bool lowMemory,
                      CudaGaussianFilter3d::TilingMode tilingMode,
                      const iolink::Vector3u32& tileSize,
                      CudaContext::Ptr cudaContext,
                      std::shared_ptr< iolink::ImageView > outputImage = nullptr );
} // namespace imagedev
