/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <iolink/ArrayX.h>
#include <memory>
#include <iolink/view/ImageView.h>

namespace imagedev
{
// Output structure of the dentalUnfolding3d function.
struct DentalUnfolding3dOutput
{
    /// The output panoramic image representing unfolded jaws. The width of this volume is defined by arch curve length. Its height is defined by the sum of the lower and upper extrusion lengths.
    std::shared_ptr< iolink::ImageView > outputPanoramicImage;
    /// The output panoramic volume given by unfolding the surface defined by the arch mesh. The width of this volume is defined by arch curve length. Its height is defined by the sum of the lower and upper extrusion lengths. Its thickness is defined by the volume thickness.
    std::shared_ptr< iolink::ImageView > outputPanoramicVolume;
    /// The quadrilateral mesh forming the surface fitted to the dental wall used to generate the output panoramic This mesh is represented as a three-dimensional array storing the coordinates of each vertex.
    std::shared_ptr<iolink::ArrayXd> inputMesh;
    /// Optional direction vectors used to define the Z axis of the output volume.
    std::shared_ptr<iolink::ArrayXd> inputDirections;
};

/// Extracts a flattened volume and image from an input three-dimensional image mapped to a surface mesh.
class IMAGEDEV_CPP_API DentalUnfolding3d final : public GenericAlgorithm
{
public:

    // Command constructor.
    DentalUnfolding3d();


    /// Gets the inputImage parameter.
    /// The input 3D CBCT image of the head (LP orientation) to unfold.
    std::shared_ptr< iolink::ImageView > inputImage() const;
    /// Sets the inputImage parameter.
    /// The input 3D CBCT image of the head (LP orientation) to unfold.
    void setInputImage( std::shared_ptr< iolink::ImageView > inputImage );

    /// Gets the inputMesh parameter.
    /// The quadrilateral mesh forming the surface fitted to the dental wall used to generate the output panoramic This mesh is represented as a three-dimensional array storing the coordinates of each vertex.
    std::shared_ptr<iolink::ArrayXd> inputMesh() const;
    /// Sets the inputMesh parameter.
    /// The quadrilateral mesh forming the surface fitted to the dental wall used to generate the output panoramic This mesh is represented as a three-dimensional array storing the coordinates of each vertex.
    void setInputMesh( std::shared_ptr<iolink::ArrayXd> inputMesh );

    /// Gets the inputDirections parameter.
    /// Optional direction vectors used to define the Z axis of the output volume.
    std::shared_ptr<iolink::ArrayXd> inputDirections() const;
    /// Sets the inputDirections parameter.
    /// Optional direction vectors used to define the Z axis of the output volume.
    void setInputDirections( std::shared_ptr<iolink::ArrayXd> inputDirections );

    /// Gets the volumeThickness parameter.
    /// The thickness of the output panoramic volume. This thickness is expressed in mm.
    double volumeThickness() const;
    /// Sets the volumeThickness parameter.
    /// The thickness of the output panoramic volume. This thickness is expressed in mm.
    void setVolumeThickness( const double& volumeThickness );

    /// Gets the slabThickness parameter.
    /// Set the number of slices of the panoramic volume used to build the output panoramic image.
    uint32_t slabThickness() const;
    /// Sets the slabThickness parameter.
    /// Set the number of slices of the panoramic volume used to build the output panoramic image.
    void setSlabThickness( const uint32_t& slabThickness );

    /// Gets the sharpening parameter.
    /// The extrusion length of dental arch curve in the direction of head towards feet. This length is expressed in mm.
    double sharpening() const;
    /// Sets the sharpening parameter.
    /// The extrusion length of dental arch curve in the direction of head towards feet. This length is expressed in mm.
    void setSharpening( const double& sharpening );

    /// Gets the outputPanoramicImage parameter.
    /// The output panoramic image representing unfolded jaws. The width of this volume is defined by arch curve length. Its height is defined by the sum of the lower and upper extrusion lengths.
    std::shared_ptr< iolink::ImageView > outputPanoramicImage() const;
    /// Sets the outputPanoramicImage parameter.
    /// The output panoramic image representing unfolded jaws. The width of this volume is defined by arch curve length. Its height is defined by the sum of the lower and upper extrusion lengths.
    void setOutputPanoramicImage( std::shared_ptr< iolink::ImageView > outputPanoramicImage );

    /// Gets the outputPanoramicVolume parameter.
    /// The output panoramic volume given by unfolding the surface defined by the arch mesh. The width of this volume is defined by arch curve length. Its height is defined by the sum of the lower and upper extrusion lengths. Its thickness is defined by the volume thickness.
    std::shared_ptr< iolink::ImageView > outputPanoramicVolume() const;
    /// Sets the outputPanoramicVolume parameter.
    /// The output panoramic volume given by unfolding the surface defined by the arch mesh. The width of this volume is defined by arch curve length. Its height is defined by the sum of the lower and upper extrusion lengths. Its thickness is defined by the volume thickness.
    void setOutputPanoramicVolume( std::shared_ptr< iolink::ImageView > outputPanoramicVolume );

    // Method to launch the command.
    void execute();

};

/// Extracts a flattened volume and image from an input three-dimensional image mapped to a surface mesh.
/// @param inputImage The input 3D CBCT image of the head (LP orientation) to unfold.
/// @param inputMesh The quadrilateral mesh forming the surface fitted to the dental wall used to generate the output panoramic This mesh is represented as a three-dimensional array storing the coordinates of each vertex.
/// @param inputDirections Optional direction vectors used to define the Z axis of the output volume.
/// @param volumeThickness The thickness of the output panoramic volume. This thickness is expressed in mm.
/// @param slabThickness Set the number of slices of the panoramic volume used to build the output panoramic image.
/// @param sharpening The extrusion length of dental arch curve in the direction of head towards feet. This length is expressed in mm.
/// @param outputPanoramicImage The output panoramic image representing unfolded jaws. The width of this volume is defined by arch curve length. Its height is defined by the sum of the lower and upper extrusion lengths.
/// @param outputPanoramicVolume The output panoramic volume given by unfolding the surface defined by the arch mesh. The width of this volume is defined by arch curve length. Its height is defined by the sum of the lower and upper extrusion lengths. Its thickness is defined by the volume thickness.
/// @return Returns a DentalUnfolding3dOutput structure containing the outputPanoramicImage and outputPanoramicVolume output parameters.
IMAGEDEV_CPP_API 
DentalUnfolding3dOutput
dentalUnfolding3d( std::shared_ptr< iolink::ImageView > inputImage,
                   std::shared_ptr<iolink::ArrayXd> inputMesh,
                   std::shared_ptr<iolink::ArrayXd> inputDirections,
                   double volumeThickness,
                   uint32_t slabThickness,
                   double sharpening,
                   std::shared_ptr< iolink::ImageView > outputPanoramicImage = nullptr,
                   std::shared_ptr< iolink::ImageView > outputPanoramicVolume = nullptr );
} // namespace imagedev
