/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <iolink/view/ImageView.h>

namespace imagedev
{
/// Computes the distance map of objects in a two-dimensional binary image.
class IMAGEDEV_CPP_API DistanceMap2d final : public GenericAlgorithm
{
public:
    /// The mode defining where the distance field is computed.
    enum MappingMode
    {
    /// Distances are computed for pixels inside the objects (having an intensity equal to 1).
        INSIDE = 0,
    /// Distances are computed for pixels outside of objects (having an intensity equal to 0).
        OUTSIDE
    };
    /// The mode defining the border conditions.
    enum BorderCondition
    {
    /// Pixels outside of the image are considered to have the value 0.
        ZERO = 0,
    /// Pixels outside of the image are considered as an extrapolation of the image pixels with a mirror border condition.
        MIRROR
    };
    /// The output image type to provide.
    enum OutputType
    {
    /// The output image type is 16-bit signed integer.
        SIGNED_INTEGER_16_BIT = 0,
    /// The output image type is 32-bit float.
        FLOAT_32_BIT
    };

    // Command constructor.
    DistanceMap2d();


    /// Gets the inputBinaryImage parameter.
    /// The binary input image.
    std::shared_ptr< iolink::ImageView > inputBinaryImage() const;
    /// Sets the inputBinaryImage parameter.
    /// The binary input image.
    void setInputBinaryImage( std::shared_ptr< iolink::ImageView > inputBinaryImage );

    /// Gets the mappingMode parameter.
    /// The mode defining where the distance field is computed.
    DistanceMap2d::MappingMode mappingMode() const;
    /// Sets the mappingMode parameter.
    /// The mode defining where the distance field is computed.
    void setMappingMode( const DistanceMap2d::MappingMode& mappingMode );

    /// Gets the borderCondition parameter.
    /// The mode defining the border conditions.
    DistanceMap2d::BorderCondition borderCondition() const;
    /// Sets the borderCondition parameter.
    /// The mode defining the border conditions.
    void setBorderCondition( const DistanceMap2d::BorderCondition& borderCondition );

    /// Gets the edgeDistance parameter.
    /// The distance weight between 2 pixels having an edge connection.
    double edgeDistance() const;
    /// Sets the edgeDistance parameter.
    /// The distance weight between 2 pixels having an edge connection.
    void setEdgeDistance( const double& edgeDistance );

    /// Gets the cornerDistance parameter.
    /// The distance weight between 2 pixels having a corner connection.
    double cornerDistance() const;
    /// Sets the cornerDistance parameter.
    /// The distance weight between 2 pixels having a corner connection.
    void setCornerDistance( const double& cornerDistance );

    /// Gets the outputType parameter.
    /// The output image type to provide.
    DistanceMap2d::OutputType outputType() const;
    /// Sets the outputType parameter.
    /// The output image type to provide.
    void setOutputType( const DistanceMap2d::OutputType& outputType );

    /// Gets the outputMapImage parameter.
    /// The output distance map image.
    std::shared_ptr< iolink::ImageView > outputMapImage() const;
    /// Sets the outputMapImage parameter.
    /// The output distance map image.
    void setOutputMapImage( std::shared_ptr< iolink::ImageView > outputMapImage );

    // Method to launch the command.
    void execute();

};

/// Computes the distance map of objects in a two-dimensional binary image.
/// @param inputBinaryImage The binary input image.
/// @param mappingMode The mode defining where the distance field is computed.
/// @param borderCondition The mode defining the border conditions.
/// @param edgeDistance The distance weight between 2 pixels having an edge connection.
/// @param cornerDistance The distance weight between 2 pixels having a corner connection.
/// @param outputType The output image type to provide.
/// @param outputMapImage The output distance map image.
/// @return Returns the outputMapImage output parameter.
IMAGEDEV_CPP_API 
std::shared_ptr< iolink::ImageView >
distanceMap2d( std::shared_ptr< iolink::ImageView > inputBinaryImage,
               DistanceMap2d::MappingMode mappingMode,
               DistanceMap2d::BorderCondition borderCondition,
               double edgeDistance,
               double cornerDistance,
               DistanceMap2d::OutputType outputType,
               std::shared_ptr< iolink::ImageView > outputMapImage = nullptr );
} // namespace imagedev
