/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <iolink/view/ImageView.h>
#include <ImageDev/Utilities/Deprecation.h>

namespace imagedev
{
// Output structure of the gradientOperator3d function.
struct GradientOperator3dOutput
{
    /// The X-gradient output image.
    std::shared_ptr< iolink::ImageView > outputImageX;
    /// The Y-gradient output image.
    std::shared_ptr< iolink::ImageView > outputImageY;
    /// The Z-gradient output image.
    std::shared_ptr< iolink::ImageView > outputImageZ;
    /// The gradient amplitude output image.
    std::shared_ptr< iolink::ImageView > outputAmplitudeImage;
};

/// Provides different algorithms to perform an edge detection based on the first derivative of a three-dimensional image.
class IMAGEDEV_CPP_API GradientOperator3d final : public GenericAlgorithm
{
public:
    /// The gradient operator to apply.
    enum GradientOperator
    {
    /// The gradient is computed using the Canny-Deriche algorithm. Supported gradient output modes : AMPLITUDE_MAXIMUM, and X_Y_AND_Z_GRADIENTS.
        CANNY_DERICHE = 0,
    /// The gradient is computed using the Canny algorithm. Supported gradient output modes : AMPLITUDE_MAXIMUM.
        CANNY,
    /// The gradient is computed using the Gaussian algorithm. @BRSupported gradient output modes : AMPLITUDE_MAXIMUM and X_Y_AND_Z_GRADIENTS.
        GAUSSIAN,
    /// The gradient is computed using the Prewitt algorithm. Supported gradient output modes : AMPLITUDE_MAXIMUM and X_Y_AND_Z_GRADIENTS.
        PREWITT,
    /// The gradient is computed using the Sobel algorithm. Supported gradient output modes : AMPLITUDE_MAXIMUM and X_Y_AND_Z_GRADIENTS.
        SOBEL
    };
    /// Select an output mode.
    enum GradientMode
    {
    /// This option computes the gradient maximal amplitude. Only the outputAmplitudeImage output is set using this mode.
        AMPLITUDE_MAXIMUM = 0,
    /// This option computes gradient in X, Y and Z directions. outputGradientXImage, outGradientYImage and outputGradientZImage outputs are set using this mode.
        X_Y_AND_Z_GRADIENTS,
    /// This option computes the Euclidean amplitude gradient. Only the outputAmplitudeImage output is set using this mode.
        AMPLITUDE_EUCLIDEAN
    };

    // Command constructor.
    IMAGEDEV_DEPRECATED( "This command is deprecated, it will be removed in ImageDev 2025.1. You can use GradientVector3d or GradientMagnitude3d instead.") GradientOperator3d();


    /// Gets the inputImage parameter.
    /// The input image.
    std::shared_ptr< iolink::ImageView > inputImage() const;
    /// Sets the inputImage parameter.
    /// The input image.
    void setInputImage( std::shared_ptr< iolink::ImageView > inputImage );

    /// Gets the gradientOperator parameter.
    /// The gradient operator to apply.
    GradientOperator3d::GradientOperator gradientOperator() const;
    /// Sets the gradientOperator parameter.
    /// The gradient operator to apply.
    void setGradientOperator( const GradientOperator3d::GradientOperator& gradientOperator );

    /// Gets the gradientMode parameter.
    /// Select an output mode.
    GradientOperator3d::GradientMode gradientMode() const;
    /// Sets the gradientMode parameter.
    /// Select an output mode.
    void setGradientMode( const GradientOperator3d::GradientMode& gradientMode );

    /// Gets the smoothingFactor parameter.
    /// The smoothing factor defines the gradient sharpness. It is only used with Canny Deriche, and Gaussian operators. It has a totally different meaning depending on the selected gradient operator. Its value must be between 0 and 100.
    double smoothingFactor() const;
    /// Sets the smoothingFactor parameter.
    /// The smoothing factor defines the gradient sharpness. It is only used with Canny Deriche, and Gaussian operators. It has a totally different meaning depending on the selected gradient operator. Its value must be between 0 and 100.
    void setSmoothingFactor( const double& smoothingFactor );

    /// Gets the outputImageX parameter.
    /// The X-gradient output image.
    std::shared_ptr< iolink::ImageView > outputImageX() const;
    /// Sets the outputImageX parameter.
    /// The X-gradient output image.
    void setOutputImageX( std::shared_ptr< iolink::ImageView > outputImageX );

    /// Gets the outputImageY parameter.
    /// The Y-gradient output image.
    std::shared_ptr< iolink::ImageView > outputImageY() const;
    /// Sets the outputImageY parameter.
    /// The Y-gradient output image.
    void setOutputImageY( std::shared_ptr< iolink::ImageView > outputImageY );

    /// Gets the outputImageZ parameter.
    /// The Z-gradient output image.
    std::shared_ptr< iolink::ImageView > outputImageZ() const;
    /// Sets the outputImageZ parameter.
    /// The Z-gradient output image.
    void setOutputImageZ( std::shared_ptr< iolink::ImageView > outputImageZ );

    /// Gets the outputAmplitudeImage parameter.
    /// The gradient amplitude output image.
    std::shared_ptr< iolink::ImageView > outputAmplitudeImage() const;
    /// Sets the outputAmplitudeImage parameter.
    /// The gradient amplitude output image.
    void setOutputAmplitudeImage( std::shared_ptr< iolink::ImageView > outputAmplitudeImage );

    // Method to launch the command.
    void execute();

};

/// Provides different algorithms to perform an edge detection based on the first derivative of a three-dimensional image.
/// @param inputImage The input image.
/// @param gradientOperator The gradient operator to apply.
/// @param gradientMode Select an output mode.
/// @param smoothingFactor The smoothing factor defines the gradient sharpness. It is only used with Canny Deriche, and Gaussian operators. It has a totally different meaning depending on the selected gradient operator. Its value must be between 0 and 100.
/// @param outputImageX The X-gradient output image.
/// @param outputImageY The Y-gradient output image.
/// @param outputImageZ The Z-gradient output image.
/// @param outputAmplitudeImage The gradient amplitude output image.
/// @return Returns a GradientOperator3dOutput structure containing the outputImageX, outputImageY, outputImageZ and outputAmplitudeImage output parameters.
IMAGEDEV_CPP_API 
GradientOperator3dOutput
gradientOperator3d( std::shared_ptr< iolink::ImageView > inputImage,
                    GradientOperator3d::GradientOperator gradientOperator,
                    GradientOperator3d::GradientMode gradientMode,
                    double smoothingFactor,
                    std::shared_ptr< iolink::ImageView > outputImageX = nullptr,
                    std::shared_ptr< iolink::ImageView > outputImageY = nullptr,
                    std::shared_ptr< iolink::ImageView > outputImageZ = nullptr,
                    std::shared_ptr< iolink::ImageView > outputAmplitudeImage = nullptr );
} // namespace imagedev
