/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <iolink/Vector.h>
#include <iolink/view/ImageView.h>
#include <ImageDev/Data/GlobalMeasurements/LocalMaximaMsr.h>

namespace imagedev
{
/// Extracts the local maxima of an image using a three-dimensional neighborhood analysis.
class IMAGEDEV_CPP_API LocalMaxima3d final : public GenericAlgorithm
{
public:
    /// The precision of the maxima localization.
    enum Precision
    {
    /// No sub-pixel approximation.
        PIXEL = 0,
    /// Sub-pixel approximation determined by parabola fitting.
        SUB_PIXEL
    };

    // Command constructor.
    LocalMaxima3d();


    /// Gets the inputGrayImage parameter.
    /// The input grayscale image
    std::shared_ptr< iolink::ImageView > inputGrayImage() const;
    /// Sets the inputGrayImage parameter.
    /// The input grayscale image
    void setInputGrayImage( std::shared_ptr< iolink::ImageView > inputGrayImage );

    /// Gets the localWindow parameter.
    /// The neighborhood size in pixels, along the X, Y, and Z directions.
    iolink::Vector3i32 localWindow() const;
    /// Sets the localWindow parameter.
    /// The neighborhood size in pixels, along the X, Y, and Z directions.
    void setLocalWindow( const iolink::Vector3i32& localWindow );

    /// Gets the thresholdRange parameter.
    /// The threshold value higher than which maxima are retained.
    double thresholdRange() const;
    /// Sets the thresholdRange parameter.
    /// The threshold value higher than which maxima are retained.
    void setThresholdRange( const double& thresholdRange );

    /// Gets the precision parameter.
    /// The precision of the maxima localization.
    LocalMaxima3d::Precision precision() const;
    /// Sets the precision parameter.
    /// The precision of the maxima localization.
    void setPrecision( const LocalMaxima3d::Precision& precision );

    /// Gets the numberOfPatterns parameter.
    /// The maximum number of patterns to be detected.
    int32_t numberOfPatterns() const;
    /// Sets the numberOfPatterns parameter.
    /// The maximum number of patterns to be detected.
    void setNumberOfPatterns( const int32_t& numberOfPatterns );

    /// Gets the outputMeasurement parameter.
    /// The result object containing the maxima positions.
    LocalMaximaMsr::Ptr outputMeasurement() const;

    // Method to launch the command.
    void execute();

};

/// Extracts the local maxima of an image using a three-dimensional neighborhood analysis.
/// @param inputGrayImage The input grayscale image
/// @param localWindow The neighborhood size in pixels, along the X, Y, and Z directions.
/// @param thresholdRange The threshold value higher than which maxima are retained.
/// @param precision The precision of the maxima localization.
/// @param numberOfPatterns The maximum number of patterns to be detected.
/// @param outputMeasurement The result object containing the maxima positions.
/// @return Returns the outputMeasurement output parameter.
IMAGEDEV_CPP_API 
LocalMaximaMsr::Ptr
localMaxima3d( std::shared_ptr< iolink::ImageView > inputGrayImage,
               const iolink::Vector3i32& localWindow,
               double thresholdRange,
               LocalMaxima3d::Precision precision,
               int32_t numberOfPatterns,
               LocalMaximaMsr::Ptr outputMeasurement = nullptr );
} // namespace imagedev
