/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <iolink/view/ImageView.h>

namespace imagedev
{
/// Computes the regional maxima or minima of a three-dimensional grayscale image and marks them in a binary image.
class IMAGEDEV_CPP_API RegionalExtrema3d final : public GenericAlgorithm
{
public:
    /// The type of extrema to detect.
    enum ExtremaType
    {
    /// The regional minima are extracted from the input image.
        MINIMA = 0,
    /// The regional maxima are extracted from the input image.
        MAXIMA
    };
    /// The 3D neighborhood configuration.
    enum Neighborhood
    {
    /// The neighborhood configuration is composed of voxels with a common face with the voxel of interest.
        CONNECTIVITY_6 = 0,
    /// The neighborhood configuration is composed of voxels with at least one common edge.
        CONNECTIVITY_18,
    /// The neighborhood configuration is a full cube.
        CONNECTIVITY_26
    };

    // Command constructor.
    RegionalExtrema3d();


    /// Gets the inputImage parameter.
    /// The input grayscale image.
    std::shared_ptr< iolink::ImageView > inputImage() const;
    /// Sets the inputImage parameter.
    /// The input grayscale image.
    void setInputImage( std::shared_ptr< iolink::ImageView > inputImage );

    /// Gets the extremaType parameter.
    /// The type of extrema to detect.
    RegionalExtrema3d::ExtremaType extremaType() const;
    /// Sets the extremaType parameter.
    /// The type of extrema to detect.
    void setExtremaType( const RegionalExtrema3d::ExtremaType& extremaType );

    /// Gets the neighborhood parameter.
    /// The 3D neighborhood configuration.
    RegionalExtrema3d::Neighborhood neighborhood() const;
    /// Sets the neighborhood parameter.
    /// The 3D neighborhood configuration.
    void setNeighborhood( const RegionalExtrema3d::Neighborhood& neighborhood );

    /// Gets the outputBinaryImage parameter.
    /// The output binary image. Its dimensions are forced to the same values as the input.
    std::shared_ptr< iolink::ImageView > outputBinaryImage() const;
    /// Sets the outputBinaryImage parameter.
    /// The output binary image. Its dimensions are forced to the same values as the input.
    void setOutputBinaryImage( std::shared_ptr< iolink::ImageView > outputBinaryImage );

    // Method to launch the command.
    void execute();

};

/// Computes the regional maxima or minima of a three-dimensional grayscale image and marks them in a binary image.
/// @param inputImage The input grayscale image.
/// @param extremaType The type of extrema to detect.
/// @param neighborhood The 3D neighborhood configuration.
/// @param outputBinaryImage The output binary image. Its dimensions are forced to the same values as the input.
/// @return Returns the outputBinaryImage output parameter.
IMAGEDEV_CPP_API 
std::shared_ptr< iolink::ImageView >
regionalExtrema3d( std::shared_ptr< iolink::ImageView > inputImage,
                   RegionalExtrema3d::ExtremaType extremaType,
                   RegionalExtrema3d::Neighborhood neighborhood,
                   std::shared_ptr< iolink::ImageView > outputBinaryImage = nullptr );
} // namespace imagedev
