/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <iolink/Vector.h>
#include <iolink/view/ImageView.h>
#include <iolink/Matrix.h>

namespace imagedev
{
/// Resamples an image by defining an oriented bounding box and a geometric transform.
class IMAGEDEV_CPP_API ResampleAffine3d final : public GenericAlgorithm
{
public:
    /// The way to determine the resolution of the output image.
    enum SamplingMode
    {
    /// The voxel size is automatically computed to be as close as possible to that of the input image. This mode does not work if the transform is not the identity.
        AUTOMATIC = 0,
    /// The output image dimensions are user-defined and the output voxel size is defined accordingly.
        MANUAL
    };
    /// The method used to calculate the intensity of each pixel in the result image.
    enum InterpolationType
    {
    /// The value of the intensity is equal to the nearest intensity.
        NEAREST_NEIGHBOR = 0,
    /// The value of the intensity is linearly interpolated in all directions.
        LINEAR
    };

    // Command constructor.
    ResampleAffine3d();


    /// Gets the inputImage parameter.
    /// The input image to resample. It can have integer or floating point data type.
    std::shared_ptr< iolink::ImageView > inputImage() const;
    /// Sets the inputImage parameter.
    /// The input image to resample. It can have integer or floating point data type.
    void setInputImage( std::shared_ptr< iolink::ImageView > inputImage );

    /// Gets the boundingBoxMin parameter.
    /// The minimum X, Y, and Z coordinates of the box that defines the image domain after a rotation, in world coordinates. The output image domain corresponds to this axis-aligned bounding box, transformed by a matrix defined by the boundingBoxTransform parameter.
    iolink::Vector3d boundingBoxMin() const;
    /// Sets the boundingBoxMin parameter.
    /// The minimum X, Y, and Z coordinates of the box that defines the image domain after a rotation, in world coordinates. The output image domain corresponds to this axis-aligned bounding box, transformed by a matrix defined by the boundingBoxTransform parameter.
    void setBoundingBoxMin( const iolink::Vector3d& boundingBoxMin );

    /// Gets the boundingBoxMax parameter.
    /// The maximum X, Y, and Z coordinates of the box that defines the image domain after a rotation, in world coordinates. The output image domain corresponds to this axis-aligned bounding box, transformed by a matrix defined by the boundingBoxTransform parameter.
    iolink::Vector3d boundingBoxMax() const;
    /// Sets the boundingBoxMax parameter.
    /// The maximum X, Y, and Z coordinates of the box that defines the image domain after a rotation, in world coordinates. The output image domain corresponds to this axis-aligned bounding box, transformed by a matrix defined by the boundingBoxTransform parameter.
    void setBoundingBoxMax( const iolink::Vector3d& boundingBoxMax );

    /// Gets the boundingBoxTransform parameter.
    /// The geometric transformation defining the bounding box orientation of the output image, represented by a 4x4 matrix.
    iolink::Matrix4d boundingBoxTransform() const;
    /// Sets the boundingBoxTransform parameter.
    /// The geometric transformation defining the bounding box orientation of the output image, represented by a 4x4 matrix.
    void setBoundingBoxTransform( const iolink::Matrix4d& boundingBoxTransform );

    /// Gets the samplingMode parameter.
    /// The way to determine the resolution of the output image.
    ResampleAffine3d::SamplingMode samplingMode() const;
    /// Sets the samplingMode parameter.
    /// The way to determine the resolution of the output image.
    void setSamplingMode( const ResampleAffine3d::SamplingMode& samplingMode );

    /// Gets the imageDimensions parameter.
    /// The size in pixels of the output image in each axis direction. If the samplingMode parameter is set to AUTOMATIC, this parameter is ignored.
    iolink::Vector3u32 imageDimensions() const;
    /// Sets the imageDimensions parameter.
    /// The size in pixels of the output image in each axis direction. If the samplingMode parameter is set to AUTOMATIC, this parameter is ignored.
    void setImageDimensions( const iolink::Vector3u32& imageDimensions );

    /// Gets the interpolationType parameter.
    /// The method used to calculate the intensity of each pixel in the result image.
    ResampleAffine3d::InterpolationType interpolationType() const;
    /// Sets the interpolationType parameter.
    /// The method used to calculate the intensity of each pixel in the result image.
    void setInterpolationType( const ResampleAffine3d::InterpolationType& interpolationType );

    /// Gets the paddingValue parameter.
    /// Specifies the output value if an output voxel position is outside the bounding box of the input image.
    double paddingValue() const;
    /// Sets the paddingValue parameter.
    /// Specifies the output value if an output voxel position is outside the bounding box of the input image.
    void setPaddingValue( const double& paddingValue );

    /// Gets the transform parameter.
    /// The geometric transformation applied to the input image before interpolation. It is represented by a 4x4 matrix. It is important to note that this transformation is the inverse transformation,  which means that it maps a point from the output domain to the input domain.
    iolink::Matrix4d transform() const;
    /// Sets the transform parameter.
    /// The geometric transformation applied to the input image before interpolation. It is represented by a 4x4 matrix. It is important to note that this transformation is the inverse transformation,  which means that it maps a point from the output domain to the input domain.
    void setTransform( const iolink::Matrix4d& transform );

    /// Gets the outputImage parameter.
    /// The output image. Its dimensions are either defined by the imageDimensions parameter, if the samplingMode parameter is MANUAL, or automatically computed to match the user-defined bounding box with a resolution close to that of the input image. Its calibration is automatically adapted. Its type and interpretation are the same as the input image.
    std::shared_ptr< iolink::ImageView > outputImage() const;
    /// Sets the outputImage parameter.
    /// The output image. Its dimensions are either defined by the imageDimensions parameter, if the samplingMode parameter is MANUAL, or automatically computed to match the user-defined bounding box with a resolution close to that of the input image. Its calibration is automatically adapted. Its type and interpretation are the same as the input image.
    void setOutputImage( std::shared_ptr< iolink::ImageView > outputImage );

    // Method to launch the command.
    void execute();

};

/// Resamples an image by defining an oriented bounding box and a geometric transform.
/// @param inputImage The input image to resample. It can have integer or floating point data type.
/// @param boundingBoxMin The minimum X, Y, and Z coordinates of the box that defines the image domain after a rotation, in world coordinates. The output image domain corresponds to this axis-aligned bounding box, transformed by a matrix defined by the boundingBoxTransform parameter.
/// @param boundingBoxMax The maximum X, Y, and Z coordinates of the box that defines the image domain after a rotation, in world coordinates. The output image domain corresponds to this axis-aligned bounding box, transformed by a matrix defined by the boundingBoxTransform parameter.
/// @param boundingBoxTransform The geometric transformation defining the bounding box orientation of the output image, represented by a 4x4 matrix.
/// @param samplingMode The way to determine the resolution of the output image.
/// @param imageDimensions The size in pixels of the output image in each axis direction. If the samplingMode parameter is set to AUTOMATIC, this parameter is ignored.
/// @param interpolationType The method used to calculate the intensity of each pixel in the result image.
/// @param paddingValue Specifies the output value if an output voxel position is outside the bounding box of the input image.
/// @param transform The geometric transformation applied to the input image before interpolation. It is represented by a 4x4 matrix. It is important to note that this transformation is the inverse transformation,  which means that it maps a point from the output domain to the input domain.
/// @param outputImage The output image. Its dimensions are either defined by the imageDimensions parameter, if the samplingMode parameter is MANUAL, or automatically computed to match the user-defined bounding box with a resolution close to that of the input image. Its calibration is automatically adapted. Its type and interpretation are the same as the input image.
/// @return Returns the outputImage output parameter.
IMAGEDEV_CPP_API 
std::shared_ptr< iolink::ImageView >
resampleAffine3d( std::shared_ptr< iolink::ImageView > inputImage,
                  const iolink::Vector3d& boundingBoxMin,
                  const iolink::Vector3d& boundingBoxMax,
                  const iolink::Matrix4d& boundingBoxTransform,
                  ResampleAffine3d::SamplingMode samplingMode,
                  const iolink::Vector3u32& imageDimensions,
                  ResampleAffine3d::InterpolationType interpolationType,
                  double paddingValue,
                  const iolink::Matrix4d& transform,
                  std::shared_ptr< iolink::ImageView > outputImage = nullptr );
} // namespace imagedev
