/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <iolink/Vector.h>
#include <iolink/view/DataFrameView.h>
#include <iolink/view/ImageView.h>

namespace imagedev
{
// Output structure of the scaleSpaceLogFilter2d function.
struct ScaleSpaceLogFilter2dOutput
{
    /// The output image representing the maximum LoG response among all scale inputs. Its dimensions, calibration and interpretation are forced to the same values as the input image. Its type is forced to float.
    std::shared_ptr< iolink::ImageView > outputImage;
    /// The output image from which each intensity represents the index of the selected scale. Its dimensions, calibration and interpretation are forced to the same values as the input image. Its type is forced to 8-bit unsigned integer.
    std::shared_ptr< iolink::ImageView > outputScaleImage;
    /// The output table contains all couples of standard deviation values for X and Y axes used to compute each scale.
    std::shared_ptr< iolink::DataFrameView > scaleTable;
};

/// This filter applies a Laplacian of Gaussian (LoG) filter at different scales, to enhance circular or elliptical (blob-like) objects.
class IMAGEDEV_CPP_API ScaleSpaceLogFilter2d final : public GenericAlgorithm
{
public:
    /// The lightness type of object to detect.
    enum Lightness
    {
    /// Bright objects are extracted from a dark background.
        BRIGHT = 0,
    /// Dark objects are extracted from a bright background.
        DARK
    };

    // Command constructor.
    ScaleSpaceLogFilter2d();


    /// Gets the inputImage parameter.
    /// The input grayscale image. The image type can be integer or floating point.
    std::shared_ptr< iolink::ImageView > inputImage() const;
    /// Sets the inputImage parameter.
    /// The input grayscale image. The image type can be integer or floating point.
    void setInputImage( std::shared_ptr< iolink::ImageView > inputImage );

    /// Gets the minimumStandardDeviation parameter.
    /// The Minimum standard deviation for X and Y axes, in pixels.
    iolink::Vector2d minimumStandardDeviation() const;
    /// Sets the minimumStandardDeviation parameter.
    /// The Minimum standard deviation for X and Y axes, in pixels.
    void setMinimumStandardDeviation( const iolink::Vector2d& minimumStandardDeviation );

    /// Gets the maximumStandardDeviation parameter.
    /// The Maximum standard deviation for X and Y axes, in pixels.
    iolink::Vector2d maximumStandardDeviation() const;
    /// Sets the maximumStandardDeviation parameter.
    /// The Maximum standard deviation for X and Y axes, in pixels.
    void setMaximumStandardDeviation( const iolink::Vector2d& maximumStandardDeviation );

    /// Gets the numberOfScales parameter.
    /// The number of scales used for computing Laplacian of Gaussian filters.
    uint32_t numberOfScales() const;
    /// Sets the numberOfScales parameter.
    /// The number of scales used for computing Laplacian of Gaussian filters.
    void setNumberOfScales( const uint32_t& numberOfScales );

    /// Gets the lightness parameter.
    /// The lightness type of object to detect.
    ScaleSpaceLogFilter2d::Lightness lightness() const;
    /// Sets the lightness parameter.
    /// The lightness type of object to detect.
    void setLightness( const ScaleSpaceLogFilter2d::Lightness& lightness );

    /// Gets the selectIndexImage parameter.
    /// The index image is generated only if set to true.
    bool selectIndexImage() const;
    /// Sets the selectIndexImage parameter.
    /// The index image is generated only if set to true.
    void setSelectIndexImage( const bool& selectIndexImage );

    /// Gets the outputImage parameter.
    /// The output image representing the maximum LoG response among all scale inputs. Its dimensions, calibration and interpretation are forced to the same values as the input image. Its type is forced to float.
    std::shared_ptr< iolink::ImageView > outputImage() const;
    /// Sets the outputImage parameter.
    /// The output image representing the maximum LoG response among all scale inputs. Its dimensions, calibration and interpretation are forced to the same values as the input image. Its type is forced to float.
    void setOutputImage( std::shared_ptr< iolink::ImageView > outputImage );

    /// Gets the scaleTable parameter.
    /// The output table contains all couples of standard deviation values for X and Y axes used to compute each scale.
    std::shared_ptr< iolink::DataFrameView > scaleTable() const;
    /// Sets the scaleTable parameter.
    /// The output table contains all couples of standard deviation values for X and Y axes used to compute each scale.
    void setScaleTable( const std::shared_ptr< iolink::DataFrameView >& scaleTable );

    /// Gets the outputScaleImage parameter.
    /// The output image from which each intensity represents the index of the selected scale. Its dimensions, calibration and interpretation are forced to the same values as the input image. Its type is forced to 8-bit unsigned integer.
    std::shared_ptr< iolink::ImageView > outputScaleImage() const;
    /// Sets the outputScaleImage parameter.
    /// The output image from which each intensity represents the index of the selected scale. Its dimensions, calibration and interpretation are forced to the same values as the input image. Its type is forced to 8-bit unsigned integer.
    void setOutputScaleImage( std::shared_ptr< iolink::ImageView > outputScaleImage );

    // Method to launch the command.
    void execute();

};

/// This filter applies a Laplacian of Gaussian (LoG) filter at different scales, to enhance circular or elliptical (blob-like) objects.
/// @param inputImage The input grayscale image. The image type can be integer or floating point.
/// @param minimumStandardDeviation The Minimum standard deviation for X and Y axes, in pixels.
/// @param maximumStandardDeviation The Maximum standard deviation for X and Y axes, in pixels.
/// @param numberOfScales The number of scales used for computing Laplacian of Gaussian filters.
/// @param lightness The lightness type of object to detect.
/// @param selectIndexImage The index image is generated only if set to true.
/// @param outputImage The output image representing the maximum LoG response among all scale inputs. Its dimensions, calibration and interpretation are forced to the same values as the input image. Its type is forced to float.
/// @param scaleTable The output table contains all couples of standard deviation values for X and Y axes used to compute each scale.
/// @param outputScaleImage The output image from which each intensity represents the index of the selected scale. Its dimensions, calibration and interpretation are forced to the same values as the input image. Its type is forced to 8-bit unsigned integer.
/// @return Returns a ScaleSpaceLogFilter2dOutput structure containing the outputImage, scaleTable and outputScaleImage output parameters.
IMAGEDEV_CPP_API 
ScaleSpaceLogFilter2dOutput
scaleSpaceLogFilter2d( std::shared_ptr< iolink::ImageView > inputImage,
                       const iolink::Vector2d& minimumStandardDeviation,
                       const iolink::Vector2d& maximumStandardDeviation,
                       uint32_t numberOfScales,
                       ScaleSpaceLogFilter2d::Lightness lightness,
                       bool selectIndexImage,
                       std::shared_ptr< iolink::ImageView > outputImage = nullptr,
                       std::shared_ptr< iolink::DataFrameView > scaleTable = nullptr,
                       std::shared_ptr< iolink::ImageView > outputScaleImage = nullptr );
} // namespace imagedev
