/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <iolink/Vector.h>
#include <iolink/view/ImageView.h>
#include <ImageDev/Data/GlobalMeasurements/SsimValueMsr.h>

namespace imagedev
{
/// Computes the mean structural similarity (SSIM) index between two three-dimensional images.
class IMAGEDEV_CPP_API SsimValue3d final : public GenericAlgorithm
{
public:
    /// The way to consider the dynamic range used to compute the regularization constants C1 and C2 of the similarity formula.
    enum RangeMode
    {
    /// The data range is determined by the theoretical maximum and minimum values of the data type.
        DATA_TYPE_RANGE = 0,
    /// The data range is user-defined by the range parameter.
        OTHER
    };

    // Command constructor.
    SsimValue3d();


    /// Gets the inputImage1 parameter.
    /// The first input image to compare.
    std::shared_ptr< iolink::ImageView > inputImage1() const;
    /// Sets the inputImage1 parameter.
    /// The first input image to compare.
    void setInputImage1( std::shared_ptr< iolink::ImageView > inputImage1 );

    /// Gets the inputImage2 parameter.
    /// The second input image to compare.
    std::shared_ptr< iolink::ImageView > inputImage2() const;
    /// Sets the inputImage2 parameter.
    /// The second input image to compare.
    void setInputImage2( std::shared_ptr< iolink::ImageView > inputImage2 );

    /// Gets the tileSize parameter.
    /// The X, Y and Z sizes, in pixels, of the tile used to compute the SSIM for each pixel.
    iolink::Vector3u32 tileSize() const;
    /// Sets the tileSize parameter.
    /// The X, Y and Z sizes, in pixels, of the tile used to compute the SSIM for each pixel.
    void setTileSize( const iolink::Vector3u32& tileSize );

    /// Gets the rangeMode parameter.
    /// The way to consider the dynamic range used to compute the regularization constants C1 and C2 of the similarity formula.
    SsimValue3d::RangeMode rangeMode() const;
    /// Sets the rangeMode parameter.
    /// The way to consider the dynamic range used to compute the regularization constants C1 and C2 of the similarity formula.
    void setRangeMode( const SsimValue3d::RangeMode& rangeMode );

    /// Gets the range parameter.
    /// The data range used to compute the constants C1 and C2 of the similarity formula. It corresponds to the distance between the maximum and minimum of the representative image intensities. It is used only if rangeMode parameter is set to OTHER value.
    double range() const;
    /// Sets the range parameter.
    /// The data range used to compute the constants C1 and C2 of the similarity formula. It corresponds to the distance between the maximum and minimum of the representative image intensities. It is used only if rangeMode parameter is set to OTHER value.
    void setRange( const double& range );

    /// Gets the k1 parameter.
    /// The small weight used to compute the regularization constant C1 of the similarity formula.
    double k1() const;
    /// Sets the k1 parameter.
    /// The small weight used to compute the regularization constant C1 of the similarity formula.
    void setK1( const double& k1 );

    /// Gets the k2 parameter.
    /// The small weight used to compute the regularization constant C2 of the similarity formula.
    double k2() const;
    /// Sets the k2 parameter.
    /// The small weight used to compute the regularization constant C2 of the similarity formula.
    void setK2( const double& k2 );

    /// Gets the outputMeasurement parameter.
    /// The output measurement results containing the SSIM value.
    SsimValueMsr::Ptr outputMeasurement() const;

    // Method to launch the command.
    void execute();

};

/// Computes the mean structural similarity (SSIM) index between two three-dimensional images.
/// @param inputImage1 The first input image to compare.
/// @param inputImage2 The second input image to compare.
/// @param tileSize The X, Y and Z sizes, in pixels, of the tile used to compute the SSIM for each pixel.
/// @param rangeMode The way to consider the dynamic range used to compute the regularization constants C1 and C2 of the similarity formula.
/// @param range The data range used to compute the constants C1 and C2 of the similarity formula. It corresponds to the distance between the maximum and minimum of the representative image intensities. It is used only if rangeMode parameter is set to OTHER value.
/// @param k1 The small weight used to compute the regularization constant C1 of the similarity formula.
/// @param k2 The small weight used to compute the regularization constant C2 of the similarity formula.
/// @param outputMeasurement The output measurement results containing the SSIM value.
/// @return Returns the outputMeasurement output parameter.
IMAGEDEV_CPP_API 
SsimValueMsr::Ptr
ssimValue3d( std::shared_ptr< iolink::ImageView > inputImage1,
             std::shared_ptr< iolink::ImageView > inputImage2,
             const iolink::Vector3u32& tileSize,
             SsimValue3d::RangeMode rangeMode,
             double range,
             double k1,
             double k2,
             SsimValueMsr::Ptr outputMeasurement = nullptr );
} // namespace imagedev
