/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <iolink/Vector.h>
#include <iolink/ArrayX.h>
#include <memory>
#include <iolink/view/ImageView.h>

namespace imagedev
{
/// Extracts a flattened 3D image from an input 3D image mapped to a surface mesh.
class IMAGEDEV_CPP_API SurfaceUnfolding3d final : public GenericAlgorithm
{
public:
    /// The interpolation mode.
    enum InterpolationType
    {
    /// The value of the intensity is equal to the nearest intensity.
        NEAREST_NEIGHBOR = 0,
    /// The value of the intensity is linearly interpolated in all directions.
        LINEAR
    };

    // Command constructor.
    SurfaceUnfolding3d();


    /// Gets the inputImage parameter.
    /// The input 3D image.
    std::shared_ptr< iolink::ImageView > inputImage() const;
    /// Sets the inputImage parameter.
    /// The input 3D image.
    void setInputImage( std::shared_ptr< iolink::ImageView > inputImage );

    /// Gets the inputSurface parameter.
    /// The input surface as a starting point for the process.
    std::shared_ptr<iolink::ArrayXd> inputSurface() const;
    /// Sets the inputSurface parameter.
    /// The input surface as a starting point for the process.
    void setInputSurface( std::shared_ptr<iolink::ArrayXd> inputSurface );

    /// Gets the directions parameter.
    /// Optional direction vectors used to unfold the surface.
    std::shared_ptr<iolink::ArrayXd> directions() const;
    /// Sets the directions parameter.
    /// Optional direction vectors used to unfold the surface.
    void setDirections( std::shared_ptr<iolink::ArrayXd> directions );

    /// Gets the thickness parameter.
    /// The output image thickness expressed in calibration unit.
    double thickness() const;
    /// Sets the thickness parameter.
    /// The output image thickness expressed in calibration unit.
    void setThickness( const double& thickness );

    /// Gets the size parameter.
    /// The dimensions of the output image in number of voxels, respectively in the X, Y, and Z directions.
    iolink::Vector3u32 size() const;
    /// Sets the size parameter.
    /// The dimensions of the output image in number of voxels, respectively in the X, Y, and Z directions.
    void setSize( const iolink::Vector3u32& size );

    /// Gets the interpolationType parameter.
    /// The interpolation mode.
    SurfaceUnfolding3d::InterpolationType interpolationType() const;
    /// Sets the interpolationType parameter.
    /// The interpolation mode.
    void setInterpolationType( const SurfaceUnfolding3d::InterpolationType& interpolationType );

    /// Gets the paddingValue parameter.
    /// The default output value.
    double paddingValue() const;
    /// Sets the paddingValue parameter.
    /// The default output value.
    void setPaddingValue( const double& paddingValue );

    /// Gets the outputImage parameter.
    /// The output image.
    std::shared_ptr< iolink::ImageView > outputImage() const;
    /// Sets the outputImage parameter.
    /// The output image.
    void setOutputImage( std::shared_ptr< iolink::ImageView > outputImage );

    // Method to launch the command.
    void execute();

};

/// Extracts a flattened 3D image from an input 3D image mapped to a surface mesh.
/// @param inputImage The input 3D image.
/// @param inputSurface The input surface as a starting point for the process.
/// @param directions Optional direction vectors used to unfold the surface.
/// @param thickness The output image thickness expressed in calibration unit.
/// @param size The dimensions of the output image in number of voxels, respectively in the X, Y, and Z directions.
/// @param interpolationType The interpolation mode.
/// @param paddingValue The default output value.
/// @param outputImage The output image.
/// @return Returns the outputImage output parameter.
IMAGEDEV_CPP_API 
std::shared_ptr< iolink::ImageView >
surfaceUnfolding3d( std::shared_ptr< iolink::ImageView > inputImage,
                    std::shared_ptr<iolink::ArrayXd> inputSurface,
                    std::shared_ptr<iolink::ArrayXd> directions,
                    double thickness,
                    const iolink::Vector3u32& size,
                    SurfaceUnfolding3d::InterpolationType interpolationType,
                    double paddingValue,
                    std::shared_ptr< iolink::ImageView > outputImage = nullptr );
} // namespace imagedev
