/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <iolink/view/ImageView.h>
#include <ImageDev/Data/Model/TextureClassificationModel.h>

namespace imagedev
{
// Output structure of the textureClassificationApply function.
struct TextureClassificationApplyOutput
{
    /// The output label image representing the texture classification result. Its dimensions and type are forced to the same values as the training input.
    std::shared_ptr< iolink::ImageView > outputLabelImage;
    /// The output map image. Its dimensions are forced to the same values as the training image. In CLASS_DISTANCE mode, its number of channels is equal to the number of classes defined in the training image. Its data type is forced to floating point.
    std::shared_ptr< iolink::ImageView > outputMapImage;
};

/// Classifies all pixels of an image using a trained texture model.
class IMAGEDEV_CPP_API TextureClassificationApply final : public GenericAlgorithm
{
public:
    /// The type of uncertainty map image to compute.
    enum OutputMapType
    {
    /// The uncertainty map represents the Mahalanobis distance to the class selected by the classification. The closer to 0 this metric is, the more confident the classification.
        CLOSEST_DISTANCE = 0,
    /// The uncertainty map represents the Mahalanobis distance to the class selected by the classification (d1) weighted by the gap with the second closest distance (d2). The smaller this metric is, the more confident and less ambiguous the classification.
        RELATIVE_DISTANCE,
    /// The uncertainty map is a multichannel image where each channel represents the distance to the corresponding class.
        CLASS_DISTANCE,
    /// No uncertainty map is computed.
        NONE
    };

    // Command constructor.
    TextureClassificationApply();


    /// Gets the inputImage parameter.
    /// The input grayscale image.
    std::shared_ptr< iolink::ImageView > inputImage() const;
    /// Sets the inputImage parameter.
    /// The input grayscale image.
    void setInputImage( std::shared_ptr< iolink::ImageView > inputImage );

    /// Gets the inputModel parameter.
    /// The input texture classification model, previously trained.
    TextureClassificationModel::Ptr inputModel() const;
    /// Sets the inputModel parameter.
    /// The input texture classification model, previously trained.
    void setInputModel( const TextureClassificationModel::Ptr& inputModel );

    /// Gets the outputMapType parameter.
    /// The type of uncertainty map image to compute.
    TextureClassificationApply::OutputMapType outputMapType() const;
    /// Sets the outputMapType parameter.
    /// The type of uncertainty map image to compute.
    void setOutputMapType( const TextureClassificationApply::OutputMapType& outputMapType );

    /// Gets the outputLabelImage parameter.
    /// The output label image representing the texture classification result. Its dimensions and type are forced to the same values as the training input.
    std::shared_ptr< iolink::ImageView > outputLabelImage() const;
    /// Sets the outputLabelImage parameter.
    /// The output label image representing the texture classification result. Its dimensions and type are forced to the same values as the training input.
    void setOutputLabelImage( std::shared_ptr< iolink::ImageView > outputLabelImage );

    /// Gets the outputMapImage parameter.
    /// The output map image. Its dimensions are forced to the same values as the training image. In CLASS_DISTANCE mode, its number of channels is equal to the number of classes defined in the training image. Its data type is forced to floating point.
    std::shared_ptr< iolink::ImageView > outputMapImage() const;
    /// Sets the outputMapImage parameter.
    /// The output map image. Its dimensions are forced to the same values as the training image. In CLASS_DISTANCE mode, its number of channels is equal to the number of classes defined in the training image. Its data type is forced to floating point.
    void setOutputMapImage( std::shared_ptr< iolink::ImageView > outputMapImage );

    // Method to launch the command.
    void execute();

};

/// Classifies all pixels of an image using a trained texture model.
/// @param inputImage The input grayscale image.
/// @param inputModel The input texture classification model, previously trained.
/// @param outputMapType The type of uncertainty map image to compute.
/// @param outputLabelImage The output label image representing the texture classification result. Its dimensions and type are forced to the same values as the training input.
/// @param outputMapImage The output map image. Its dimensions are forced to the same values as the training image. In CLASS_DISTANCE mode, its number of channels is equal to the number of classes defined in the training image. Its data type is forced to floating point.
/// @return Returns a TextureClassificationApplyOutput structure containing the outputLabelImage and outputMapImage output parameters.
IMAGEDEV_CPP_API 
TextureClassificationApplyOutput
textureClassificationApply( std::shared_ptr< iolink::ImageView > inputImage,
                            TextureClassificationModel::Ptr inputModel,
                            TextureClassificationApply::OutputMapType outputMapType,
                            std::shared_ptr< iolink::ImageView > outputLabelImage = nullptr,
                            std::shared_ptr< iolink::ImageView > outputMapImage = nullptr );
} // namespace imagedev
