/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <iolink/Vector.h>
#include <ImageDev/Data/Model/TextureClassificationModel.h>

namespace imagedev
{
/// Creates a new object model for performing a texture classification.
class IMAGEDEV_CPP_API TextureClassificationCreate final : public GenericAlgorithm
{
public:
    /// The computation mode: 2D or 3D.
    enum ComputeMode
    {
    /// Computation as 2D data.
        MODE_2D = 0,
    /// Computation as 3D data.
        MODE_3D
    };
    /// The groups of textural features to compute. This list defines all the textural attributes proposed for performing the classification.
    enum FeatureGroup
    {
    /// Features based on co-occurrence matrices. One feature is extracted from each co-occurrence vector.
        DIRECTIONAL_COOCCURRENCE = 1,
    /// Features based on co-occurrence matrices. Three statistical features are extracted from all vectors.
        ROTATION_INVARIANT_COOCCURRENCE = 2,
    /// Features based on first order statistics that are not computed using a histogram.
        FIRST_ORDER_STATISTICS = 4,
    /// Features based on histogram statistics, including histogram quantiles.
        HISTOGRAM_STATISTICS = 8,
    /// Feature based on the intensity value of the input image.
        INTENSITY = 16
    };
    /// The shape of the co-occurrence texton (the pattern defined by the set of co-occurrence vectors). This parameter is ignored if none of the co-occurrence feature groups is selected.
    enum CoocTextonShape
    {
    /// The set of all points associated to corners and edge centers of a square with a half side size defined by the coocTextonSize parameter.
        CUBE = 0,
    /// The set of all points located at the same euclidean distance defined by the coocTextonSize parameter from the center. This mode is recommended when a repetitive texture is mono-scale.
        SPHERE,
    /// The set of all points located at a distance less or equal to the coocTextonSize parameter from the center. This mode can be useful to classify a multi-scale repetitive texture, but may be very time consuming.
        BALL
    };

    // Command constructor.
    TextureClassificationCreate();


    /// Gets the computeMode parameter.
    /// The computation mode: 2D or 3D.
    TextureClassificationCreate::ComputeMode computeMode() const;
    /// Sets the computeMode parameter.
    /// The computation mode: 2D or 3D.
    void setComputeMode( const TextureClassificationCreate::ComputeMode& computeMode );

    /// Gets the numberOfClasses parameter.
    /// The number of classes to discriminate.
    uint32_t numberOfClasses() const;
    /// Sets the numberOfClasses parameter.
    /// The number of classes to discriminate.
    void setNumberOfClasses( const uint32_t& numberOfClasses );

    /// Gets the featureGroup parameter.
    /// The groups of textural features to compute. This list defines all the textural attributes proposed for performing the classification.
    int32_t featureGroup() const;
    /// Sets the featureGroup parameter.
    /// The groups of textural features to compute. This list defines all the textural attributes proposed for performing the classification.
    void setFeatureGroup( const int32_t& featureGroup );

    /// Gets the radiusRange parameter.
    /// The minimum and maximum radius, in pixels, of the circular neighborhoods used for computing textural features.
    iolink::Vector2u32 radiusRange() const;
    /// Sets the radiusRange parameter.
    /// The minimum and maximum radius, in pixels, of the circular neighborhoods used for computing textural features.
    void setRadiusRange( const iolink::Vector2u32& radiusRange );

    /// Gets the radiusStep parameter.
    /// The step, in pixels, used to define the set of radii between minimum and maximum. The maximum radius is systematically added to the radius list.
    uint32_t radiusStep() const;
    /// Sets the radiusStep parameter.
    /// The step, in pixels, used to define the set of radii between minimum and maximum. The maximum radius is systematically added to the radius list.
    void setRadiusStep( const uint32_t& radiusStep );

    /// Gets the coocRadius parameter.
    /// The radius, in pixels, of the circular neighborhood used by the co-occurrence features. This parameter is ignored if none of the co-occurrence feature groups is selected.
    uint32_t coocRadius() const;
    /// Sets the coocRadius parameter.
    /// The radius, in pixels, of the circular neighborhood used by the co-occurrence features. This parameter is ignored if none of the co-occurrence feature groups is selected.
    void setCoocRadius( const uint32_t& coocRadius );

    /// Gets the coocTextonShape parameter.
    /// The shape of the co-occurrence texton (the pattern defined by the set of co-occurrence vectors). This parameter is ignored if none of the co-occurrence feature groups is selected.
    TextureClassificationCreate::CoocTextonShape coocTextonShape() const;
    /// Sets the coocTextonShape parameter.
    /// The shape of the co-occurrence texton (the pattern defined by the set of co-occurrence vectors). This parameter is ignored if none of the co-occurrence feature groups is selected.
    void setCoocTextonShape( const TextureClassificationCreate::CoocTextonShape& coocTextonShape );

    /// Gets the coocTextonSize parameter.
    /// The size, in pixels, of the texton shape for co-occurrence features. This parameter is ignored if none of the co-occurrence feature groups is selected.
    uint32_t coocTextonSize() const;
    /// Sets the coocTextonSize parameter.
    /// The size, in pixels, of the texton shape for co-occurrence features. This parameter is ignored if none of the co-occurrence feature groups is selected.
    void setCoocTextonSize( const uint32_t& coocTextonSize );

    /// Gets the outputModel parameter.
    /// The output texture classification model, newly created.
    TextureClassificationModel::Ptr outputModel() const;

    // Method to launch the command.
    void execute();

};

/// Creates a new object model for performing a texture classification.
/// @param computeMode The computation mode: 2D or 3D.
/// @param numberOfClasses The number of classes to discriminate.
/// @param featureGroup The groups of textural features to compute. This list defines all the textural attributes proposed for performing the classification.
/// @param radiusRange The minimum and maximum radius, in pixels, of the circular neighborhoods used for computing textural features.
/// @param radiusStep The step, in pixels, used to define the set of radii between minimum and maximum. The maximum radius is systematically added to the radius list.
/// @param coocRadius The radius, in pixels, of the circular neighborhood used by the co-occurrence features. This parameter is ignored if none of the co-occurrence feature groups is selected.
/// @param coocTextonShape The shape of the co-occurrence texton (the pattern defined by the set of co-occurrence vectors). This parameter is ignored if none of the co-occurrence feature groups is selected.
/// @param coocTextonSize The size, in pixels, of the texton shape for co-occurrence features. This parameter is ignored if none of the co-occurrence feature groups is selected.
/// @param outputModel The output texture classification model, newly created.
/// @return Returns the outputModel output parameter.
IMAGEDEV_CPP_API 
TextureClassificationModel::Ptr
textureClassificationCreate( TextureClassificationCreate::ComputeMode computeMode,
                             uint32_t numberOfClasses,
                             int32_t featureGroup,
                             const iolink::Vector2u32& radiusRange,
                             uint32_t radiusStep,
                             uint32_t coocRadius,
                             TextureClassificationCreate::CoocTextonShape coocTextonShape,
                             uint32_t coocTextonSize,
                             TextureClassificationModel::Ptr outputModel = nullptr );
} // namespace imagedev
