/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/Processing/GenericAlgorithm.h>
#include <ImageDev/ImageDevCppExports.h>
#include <ImageDev/Exception.h>
#include <iolink/view/ImageView.h>

namespace imagedev
{
// Output structure of the vectorToMagnitude2d function.
struct VectorToMagnitude2dOutput
{
    /// The amplitude output image.
    std::shared_ptr< iolink::ImageView > outputAmplitudeImage;
    /// The orientation output image.
    std::shared_ptr< iolink::ImageView > outputOrientationImage;
};

/// Computes amplitude and orientation of a 2D vector such as a gradient from its X and Y components.
class IMAGEDEV_CPP_API VectorToMagnitude2d final : public GenericAlgorithm
{
public:
    /// The output mode.
    enum OutputType
    {
    /// This option computes the amplitude as the maximum between the X and Y components. Only the outputAmplitudeImage output is set using this mode.
        AMPLITUDE_MAXIMUM = 0,
    /// This option computes the Euclidean norm of the vector. Only the outputAmplitudeImage output is set using this mode.
        AMPLITUDE_EUCLIDEAN,
    /// This option computes Euclidean norm and orientation in degrees (between -128 and +128 degrees). Both outputAmplitudeImage and outputOrientationImage outputs are set using this mode.
        AMPLITUDE_AND_ORIENTATION
    };

    // Command constructor.
    VectorToMagnitude2d();


    /// Gets the inputImageX parameter.
    /// The X vector component input image.
    std::shared_ptr< iolink::ImageView > inputImageX() const;
    /// Sets the inputImageX parameter.
    /// The X vector component input image.
    void setInputImageX( std::shared_ptr< iolink::ImageView > inputImageX );

    /// Gets the inputImageY parameter.
    /// The Y vector component input image.
    std::shared_ptr< iolink::ImageView > inputImageY() const;
    /// Sets the inputImageY parameter.
    /// The Y vector component input image.
    void setInputImageY( std::shared_ptr< iolink::ImageView > inputImageY );

    /// Gets the outputType parameter.
    /// The output mode.
    VectorToMagnitude2d::OutputType outputType() const;
    /// Sets the outputType parameter.
    /// The output mode.
    void setOutputType( const VectorToMagnitude2d::OutputType& outputType );

    /// Gets the outputAmplitudeImage parameter.
    /// The amplitude output image.
    std::shared_ptr< iolink::ImageView > outputAmplitudeImage() const;
    /// Sets the outputAmplitudeImage parameter.
    /// The amplitude output image.
    void setOutputAmplitudeImage( std::shared_ptr< iolink::ImageView > outputAmplitudeImage );

    /// Gets the outputOrientationImage parameter.
    /// The orientation output image.
    std::shared_ptr< iolink::ImageView > outputOrientationImage() const;
    /// Sets the outputOrientationImage parameter.
    /// The orientation output image.
    void setOutputOrientationImage( std::shared_ptr< iolink::ImageView > outputOrientationImage );

    // Method to launch the command.
    void execute();

};

/// Computes amplitude and orientation of a 2D vector such as a gradient from its X and Y components.
/// @param inputImageX The X vector component input image.
/// @param inputImageY The Y vector component input image.
/// @param outputType The output mode.
/// @param outputAmplitudeImage The amplitude output image.
/// @param outputOrientationImage The orientation output image.
/// @return Returns a VectorToMagnitude2dOutput structure containing the outputAmplitudeImage and outputOrientationImage output parameters.
IMAGEDEV_CPP_API 
VectorToMagnitude2dOutput
vectorToMagnitude2d( std::shared_ptr< iolink::ImageView > inputImageX,
                     std::shared_ptr< iolink::ImageView > inputImageY,
                     VectorToMagnitude2d::OutputType outputType,
                     std::shared_ptr< iolink::ImageView > outputAmplitudeImage = nullptr,
                     std::shared_ptr< iolink::ImageView > outputOrientationImage = nullptr );
} // namespace imagedev
