// ================================================================================ //
//       THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),        //
//                  AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                   //
//                                                                                  //
//     REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS      //
//     SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR      //
//                       WRITTEN AUTHORIZATION OF FEI S.A.S.                        //
//                                                                                  //
//                             RESTRICTED RIGHTS LEGEND                             //
//     USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS     //
//     WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN     //
//     SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT     //
//     CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN     //
//        TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.         //
//                                                                                  //
//                         COPYRIGHT (C) 2021 BY FEI S.A.S,                         //
//                                 BORDEAUX, FRANCE                                 //
//                               ALL RIGHTS RESERVED                                //
//                                                                                  //
//        SEE https://developer.openinventor.com/MiscFiles/EULA.pdf FOR MORE        //
// ================================================================================ //

#pragma once

#include <iolink/IOLinkAPI.h>
#include <iolink/Vector.h>

#include <iostream>

namespace iolink
{

/**
 * Enum to describe the primitive family of a data type.
 *
 * Some examples:
 *  - UINT16, UINT32, and VEC3_UINT8 are all USIGNED_INTEGER
 *  - FLOAT and DOUBLE are FLOATING_POINT
 *  - UTF8_STRING and UTF16_STRING are UNICODE_STRING
 */
enum class PrimitiveTypeId : uint16_t
{
  /** An opaque type, that the no one knows about */
  UNKNOWN = 0x0000,

  /** A classic integer number */
  UNSIGNED_INTEGER = 0x0001,

  /** An integer number, using the two complement for negative numbers */
  SIGNED_INTEGER = 0x0002,

  /** A floating point number, encoded according to IEEE 754 */
  FLOATING_POINT = 0x0003,

  /** A text string, using the Unicode coding system */
  UNICODE_STRING = 0x0004,
};

/**
 * An enumeration to describe how the data should be interpreted.
 */
enum class DataTypeInterpretation : uint32_t
{
  /** No particular interpretation, classic vectors and scalars */
  RAW = 0,

  /** A complex number.
   *
   * A DataType intance with this interpretation should always
   * have a dimension of 2. The first element will be the real part
   * and the second one the imaginary part.
   */
  COMPLEX,

  /** A square matrix.
   *
   * The dimension of DataType instances with this interpretation
   * is the number of element in that matrix. For example, in the case
   * of a square matrix of size 3, its dimension will be 9.
   *
   * Example of layout:
   *
   * | 0 | 1 | 2 |
   * | 3 | 4 | 5 |
   * | 6 | 7 | 8 |
   */
  SQUARE_MATRIX,

  /** A symmetric matrix
   *
   * The dimension of DataType instances with this interpretation
   * is the number of element in that matrix. For example, in the case
   * of a symmetric matrix of size 3, its dimension will be 6.
   *
   * Example of layout:
   *
   * | 0 | 1 | 2 |
   * |   | 3 | 4 |
   * |   |   | 5 |
   */
  SYMMETRIC_MATRIX,
};

/**
 * A collection of built-in data types.
 */
enum class DataTypeId : uint64_t
{
  UNKNOWN = 0x0000000000000000,            ///< An opaque type
  UINT8 = 0x0000000000010108,              ///< 8 bits unsigned integer
  UINT16 = 0x0000000000010110,             ///< 16 bits unsigned integer
  UINT32 = 0x0000000000010120,             ///< 32 bits unsigned integer
  UINT64 = 0x0000000000010140,             ///< 64 bits unsigned integer
  INT8 = 0x0000000000020108,               ///< 8 bits signed integer
  INT16 = 0x0000000000020110,              ///< 16 bits signed integer
  INT32 = 0x0000000000020120,              ///< 32 bits signed integer
  INT64 = 0x0000000000020140,              ///< 64 bits signed integer
  FLOAT = 0x0000000000030120,              ///< Single precision floating point
  DOUBLE = 0x0000000000030140,             ///< Double precision floating point
  UTF8_STRING = 0x0000000000040108,        ///< Unicode string encoded in UTF-8
  UTF16_STRING = 0x0000000000040110,       ///< Unicode string encoded in UTF-16
  UTF32_STRING = 0x0000000000040120,       ///< Unicode string encoded in UTF-32
  VEC2_UINT8 = 0x0000000000010208,         ///< A vector of 2 elements that are 8 bits unsigned integer
  VEC2_UINT16 = 0x0000000000010210,        ///< A vector of 2 elements that are 16 bits unsigned integer
  VEC2_UINT32 = 0x0000000000010220,        ///< A vector of 2 elements that are 32 bits unsigned integer
  VEC2_UINT64 = 0x0000000000010240,        ///< A vector of 2 elements that are 64 bits unsigned integer
  VEC2_INT8 = 0x0000000000020208,          ///< A vector of 2 elements that are 8 bits signed integer
  VEC2_INT16 = 0x0000000000020210,         ///< A vector of 2 elements that are 16 bits signed integer
  VEC2_INT32 = 0x0000000000020220,         ///< A vector of 2 elements that are 32 bits signed integer
  VEC2_INT64 = 0x0000000000020240,         ///< A vector of 2 elements that are 64 bits signed integer
  VEC2_FLOAT = 0x0000000000030220,         ///< A vector of 2 elements that are single precision floating point
  VEC2_DOUBLE = 0x0000000000030240,        ///< A vector of 2 elements that are double precision floating point
  VEC3_UINT8 = 0x0000000000010308,         ///< A vector of 2 elements that are 8 bits unsigned integer
  VEC3_UINT16 = 0x0000000000010310,        ///< A vector of 3 elements that are 16 bits unsigned integer
  VEC3_UINT32 = 0x0000000000010320,        ///< A vector of 3 elements that are 32 bits unsigned integer
  VEC3_UINT64 = 0x0000000000010340,        ///< A vector of 3 elements that are 64 bits unsigned integer
  VEC3_INT8 = 0x0000000000020308,          ///< A vector of 3 elements that are 8 bits signed integer
  VEC3_INT16 = 0x0000000000020310,         ///< A vector of 3 elements that are 16 bits signed integer
  VEC3_INT32 = 0x0000000000020320,         ///< A vector of 3 elements that are 32 bits signed integer
  VEC3_INT64 = 0x0000000000020340,         ///< A vector of 3 elements that are 64 bits signed integer
  VEC3_FLOAT = 0x0000000000030320,         ///< A vector of 3 elements that are single precision floating point
  VEC3_DOUBLE = 0x0000000000030340,        ///< A vector of 3 elements that are double precision floating point
  VEC4_UINT8 = 0x0000000000010408,         ///< A vector of 4 elements that are 8 bits unsigned integer
  VEC4_UINT16 = 0x0000000000010410,        ///< A vector of 4 elements that are 16 bits unsigned integer
  VEC4_UINT32 = 0x0000000000010420,        ///< A vector of 4 elements that are 32 bits unsigned integer
  VEC4_UINT64 = 0x0000000000010440,        ///< A vector of 4 elements that are 64 bits unsigned integer
  VEC4_INT8 = 0x0000000000020408,          ///< A vector of 4 elements that are 8 bits signed integer
  VEC4_INT16 = 0x0000000000020410,         ///< A vector of 4 elements that are 16 bits signed integer
  VEC4_INT32 = 0x0000000000020420,         ///< A vector of 4 elements that are 32 bits signed integer
  VEC4_INT64 = 0x0000000000020440,         ///< A vector of 4 elements that are 64 bits signed integer
  VEC4_FLOAT = 0x0000000000030420,         ///< A vector of 4 elements that are single precision floating point
  VEC4_DOUBLE = 0x0000000000030440,        ///< A vector of 4 elements that are double precision floating point
  COMPLEX_FLOAT = 0x0000000100030220,      ///< A single precision floating point complex number
  COMPLEX_DOUBLE = 0x0000000100030240,     ///< A double precision floating point complex number
  MATRIX2_FLOAT = 0x0000000200030420,      ///< A single precision floating point 2x2 Matrix
  MATRIX2_DOUBLE = 0x0000000200030440,     ///< A double precision floating point 2x2 Matrix
  MATRIX3_FLOAT = 0x0000000200030920,      ///< A single precision floating point 3x3 Matrix
  MATRIX3_DOUBLE = 0x0000000200030940,     ///< A double precision floating point 3x3 Matrix
  MATRIX4_FLOAT = 0x0000000200031020,      ///< A single precision floating point 4x4 Matrix
  MATRIX4_DOUBLE = 0x0000000200031040,     ///< A double precision floating point 4x4 Matrix
  SYM_MATRIX2_FLOAT = 0x0000000300030320,  ///< A single precision floating point 2x2 symmetric Matrix
  SYM_MATRIX2_DOUBLE = 0x0000000300030340, ///< A double precision floating point 2x2 symmetric Matrix
  SYM_MATRIX3_FLOAT = 0x0000000300030620,  ///< A single precision floating point 3x3 symmetric Matrix
  SYM_MATRIX3_DOUBLE = 0x0000000300030640  ///< A double precision floating point 3x3 symmetric Matrix
};

/**
 * Stores information about a data type.
 *
 * DataType is associated with an ID, with wich the type can be uniquely
 * identified. Although this ID is composed of multiple fields:
 *  - ScalarType: base type of each component.
 *  - BitDepth: bit depth for each component.
 *  - Dimension: the number of components.
 *  - Interpretation: What the DataType represents in particular.
 *
 * See [user guide](md_user_guide_docs_howto.html#data_type) for more explainations.
 *
 * These fields can have different meaning bases on the scalar type or the
 * flags. For example the bitDepth field indicate the precision for integer and
 * floating point types, and the size of a coding point in an unicode string.
 */
class IOLINK_API DataType
{

public:
  /**
   * Return scalar type in case of multi dimensionnal DataType.
   *
   * If the type is already scalar, it return itself.
   */
  static DataType extractScalarType(DataType dtype);

  /**
   * Return conventional range for given DataType.
   *
   * By convention, range for INTEGER types are classical min and max representable values.
   *
   * Range for FLOATING_POINT type is [0, 1]
   *
   * As returned range is DOUBLE precison, returned value is imprecise for INT64 and UINT64 type.
   *
   * @throw Error If no range can be computed for this type.
   */
  static Vector2d standardRange(DataType dtype);

  /**
   * Default constructor, creating an opaque type.
   *
   * @see DataTypeId::UNKNOWN
   */
  DataType() noexcept;

  /**
   * Build a DataType from an id.
   */
  explicit DataType(uint64_t value) noexcept;

  /**
   * Build a DataType from a DataTypeId.
   */
  DataType(DataTypeId value) noexcept;

  /**
   * Build a DataType from its properties.
   */
  DataType(PrimitiveTypeId primitiveType,
           size_t dimension,
           size_t bitDepth,
           DataTypeInterpretation interpretation) noexcept;

  /**
   * Build a DataType from its properties.
   */
  DataType(PrimitiveTypeId primitiveType, size_t dimension, size_t bitDepth) noexcept;

  /**
   * Checks if the type is a mathematical one.
   */
  bool isNumeric() const;

  /**
   * Checks if the type is a scalar numeric value.
   */
  bool isScalar() const;

  /**
   * Checks if the type is a vector of numeric values.
   */
  bool isVector() const;

  /**
   * Checks if this type is trivially convertible to another.
   *
   * @see DataConverter
   */
  bool isConvertibleTo(DataType other) const;

  /**
   * Return unique id representing this datatype. Can be used inside a
   * switch/case statement.
   */
  uint64_t id() const noexcept;

  /**
   * Return the bit depth of this type. (i.e. UINT8 => 8 bit depth, VEC3_UINT16 => 16 bit depth)
   */
  inline size_t bitDepth() const noexcept { return m_bitDepth; }

  /**
   * Return the number of sub-values included in this type.
   *
   * For example:
   *  - UINT8 => dimension 1
   *  - VEC3_UINT8 => dimension 3 because it contains three UINT8
   */
  inline size_t elementCount() const noexcept { return m_dimension; }

  /**
   * Return the primitive type of the current type.
   *
   * for example: UINT8 => UNSIGNED_INTEGER
   */
  inline PrimitiveTypeId primitiveType() const noexcept { return m_primitiveType; }

  /**
   * The DataType interpretation.
   *
   * @see DataTypeInterpretation
   */
  inline DataTypeInterpretation interpretation() const noexcept { return m_interpretation; }

  /**
   * Get the number of bits required to store the type.
   */
  inline size_t bitCount() const { return m_bitDepth * m_dimension; }

  /**
   * Get the number of bits required to store the type.
   *
   * If the number of bits required to store the type is not aligned to a byte,
   * the number of byte will be rounded up. For example, a 12 bits integer type
   * will have a byte count of 2.
   */
  size_t byteCount() const { return (m_bitDepth * m_dimension - 1) / 8 + 1; }

  /**
   * Return a readable version of the DataType.
   */
  std::string toString() const;

  bool operator==(DataType other) const;
  bool operator!=(DataType other) const;

private:
  uint8_t m_bitDepth;
  uint8_t m_dimension;
  PrimitiveTypeId m_primitiveType;
  DataTypeInterpretation m_interpretation;
};

inline std::ostream&
operator<<(std::ostream& os, const DataType datatype)
{
  os << datatype.toString().c_str();
  return os;
}

} // end namespace iolink
