// ================================================================================ //
//       THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),        //
//                  AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                   //
//                                                                                  //
//     REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS      //
//     SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR      //
//                       WRITTEN AUTHORIZATION OF FEI S.A.S.                        //
//                                                                                  //
//                             RESTRICTED RIGHTS LEGEND                             //
//     USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS     //
//     WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN     //
//     SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT     //
//     CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN     //
//        TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.         //
//                                                                                  //
//                         COPYRIGHT (C) 2021 BY FEI S.A.S,                         //
//                                 BORDEAUX, FRANCE                                 //
//                               ALL RIGHTS RESERVED                                //
//                                                                                  //
//        SEE https://developer.openinventor.com/MiscFiles/EULA.pdf FOR MORE        //
// ================================================================================ //

#pragma once

#include <cstdint>
#include <initializer_list>
#include <string>

#include <iolink/IOLinkAPI.h>

namespace iolink
{

/**
 * Enum used in ImageType to represent each dimension of an Image
 */
enum class ImageDimension
{
  /** represents a image width */
  COLUMN = 0,
  /** represents a image height */
  ROW = 1,
  /** represents depth for a volumetric image */
  SLICE = 2,
  /** represents channel planes of a multi-spectral image */
  CHANNEL = 3,
  /** represents the temporality for a sequence of images */
  SEQUENCE = 4,
};

// create an alias for ImageDimension
using ImageAxis = ImageDimension;

/**
 * A flag system that describes the dimensions of an image.
 * An image is firstly defined by its dimensions.
 *
 * i.e.
 * - a classic Image has COLUMN and ROW dimensions
 * - a volume has COLUMN, ROW and SLICE dimensions
 * Concerning the order of dimensions, it follows #ImageDimension values order.
 *
 */
class IOLINK_API ImageType
{
public:
  /**
   * Create an ImageType with no known dimensions.
   */
  ImageType();

  /**
   * Create an ImageType with only one given dimension.
   */
  explicit ImageType(ImageDimension dimension);

  /**
   * Create an ImageType with a list of dimensions (C array style)
   */
  explicit ImageType(const ImageDimension* dimensionsList, size_t count);

  /**
   * Create an ImageType with a list of dimensions
   */
  explicit ImageType(std::initializer_list<ImageDimension> dimensionsList);

  /**
   * Create an ImageType from an 64-bits integer
   */
  explicit ImageType(uint64_t val);

  /**
   * Checks if the ImageType has the given dimension.
   * @param dimension dimension presence to be checked
   * @return TRUE if dimension is present in the ImageType
   */
  bool has(ImageDimension dimension) const;

  /**
   * Add the given dimension to the ImageType.
   * @param dimension dimension to add in the ImageType
   */
  void add(ImageDimension dimension);

  /**
   * Give the number of dimensions in the ImageType
   */
  size_t dimensionCount() const;

  /**
   * Add all the dimensions from given ImageType to current ImageType
   * @param other ImageType whose dimensions must be added into the current ImageType
   */
  void add(ImageType other);

  /**
   * Remove the given dimension to the ImageType.
   * @param dimension dimension to remove from ImageType
   */
  void remove(ImageDimension dimension);

  /**
   * Remove the nth dimension to the ImageType.
   * @param index index of the dimension to remove
   */
  void remove(size_t index);

  /**
   * Remove all the given dimensions to the ImageType.
   * @param other ImageType whose dimensions must be removed from current ImageType
   */
  void remove(ImageType other);

  /**
   * Return the nth dimension at the given position
   *
   * e.g. in an IMAGE_SEQUENCE:
   * [COLUMN](\ref #ImageDimension.COLUMN) is the dimension returned for index = 0
   * [ROW](\ref #ImageDimension.ROW) is the dimension returned for index = 1
   * [SEQUENCE](\ref #ImageDimension.SEQUENCE) is the dimension returned for index = 2
   *
   * @throw Error if index does not exist in current ImageType
   * @param index dimension index in the ImageType
   * @return the dimension corresponding to the given index
   */
  ImageDimension dimension(size_t index) const;

  /**
   * Return the position of given dimension in the current ImageType
   *
   * e.g. in an IMAGE_SEQUENCE:
   * [COLUMN](\ref #ImageDimension.COLUMN) is the first dimension (idx = 0 is returned)
   * [ROW](\ref #ImageDimension.ROW) is the second dimension (idx = 1 is returned)
   * [SEQUENCE](\ref #ImageDimension.SEQUENCE) is the 3th dimension (idx= 2 is returned)
   *
   * @param dimension the dimension whom you want to get the index in the current ImageType
   * @return -1 if the dimension is not present, otherwise the position (index) in current ImageType
   */
  size_t indexDimension(ImageDimension dimension);

  /**
   * Equivalent to add.
   @param rhs Right operand which represents a dimension to add to current ImageType
   @return ImageType with new given dimension
   */
  ImageType operator|(ImageDimension rhs) const;

  /**
   * Equality operator
   @param other Right operand which represents a ImageType to compare to current one
   @return TRUE if ImageType is equal to current one
   */
  bool operator==(ImageType other) const;

  /**
   * Not-Equality operator
   @param other Right operand which represents a ImageType to compare to current one
   @return TRUE if ImageType is NOT equal to current one
   */
  bool operator!=(ImageType other) const;

  /**
   * Return string interpretation
   */
  std::string toString() const;

  /**
   * Return internal 64-bit integer value
   */
  uint64_t value() const;

private:
  static uint64_t dimensionFlag(ImageDimension dimension);

  uint64_t m_value;
};

/**
 * Binary Operator to combine two dimensions
 * @param lhs left operand - ImageDimension
 * @param rhs right operand - ImageDimension
 * @return a ImageType which contains dimensions from left and right operands
 */
inline ImageType
operator|(ImageDimension lhs, ImageDimension rhs)
{
  return ImageType(ImageType(lhs) | rhs);
}

// create an alias for ImageType
using AxesInterpretation = ImageType;

/**
 * Binary Operator to combine a dimension with an ImageType
 * @param lhs left operand - ImageDimension
 * @param rhs left operand - ImageType
 * @return a ImageType which contains dimensions from left operand and dimensions contained in right operand
 */
inline ImageType
operator|(ImageDimension lhs, ImageType rhs)
{
  return rhs | lhs;
}

/**
 * Enum for the image type (see ImageType)
 *
 * Each bit represents a particular dimension:
 *  - X dimension
 *  - Y dimension
 *  - Z dimension
 *  - Channel dimension
 *  - Sequence dimension
 */
class IOLINK_API ImageTypeId
{
public:
  /** When ImageType is not known or defined */
  static const ImageType UNKNOWN;
  /** Represents a 2D image ([COLUMN](\ref #ImageDimension.COLUMN), [ROW](\ref #ImageDimension.ROW)) */
  static const ImageType IMAGE;
  /** Represents a 3D image ([COLUMN](\ref #ImageDimension.COLUMN), [ROW](\ref #ImageDimension.ROW), [SLICE](\ref
   * #ImageDimension.SLICE)) */
  static const ImageType VOLUME;
  /** Represents a 3D multi-spectral image ([COLUMN](\ref #ImageDimension.COLUMN), [ROW](\ref #ImageDimension.ROW),
   * [CHANNEL](\ref #ImageDimension.CHANNEL)) */
  static const ImageType MULTISPECTRAL_IMAGE;
  /** Represents a 4D multi-spectral volume ([COLUMN](\ref #ImageDimension.COLUMN), [ROW](\ref #ImageDimension.ROW),
   * [SLICE](\ref #ImageDimension.SLICE), [CHANNEL](\ref #ImageDimension.CHANNEL)) */
  static const ImageType MULTISPECTRAL_VOLUME;
  /** Represents a 3D temporal image ([COLUMN](\ref #ImageDimension.COLUMN), [ROW](\ref #ImageDimension.ROW),
   * [SEQUENCE](\ref #ImageDimension.SEQUENCE)) */
  static const ImageType IMAGE_SEQUENCE;
  /** Represents a 4D temporal volume ([COLUMN](\ref #ImageDimension.COLUMN), [ROW](\ref #ImageDimension.ROW),
   * [SLICE](\ref #ImageDimension.SLICE), [SEQUENCE](\ref #ImageDimension.SEQUENCE)) */
  static const ImageType VOLUME_SEQUENCE;
  /** Represents a 4D temporal volume ([COLUMN](\ref #ImageDimension.COLUMN), [ROW](\ref #ImageDimension.ROW),
   * [CHANNEL](\ref #ImageDimension.CHANNEL), [SEQUENCE](\ref #ImageDimension.SEQUENCE)) */
  static const ImageType MULTISPECTRAL_IMAGE_SEQUENCE;
  /** Represents a 5D temporal volume ([COLUMN](\ref #ImageDimension.COLUMN), [ROW](\ref #ImageDimension.ROW),
   * [SLICE](\ref #ImageDimension.SLICE), [CHANNEL](\ref #ImageDimension.CHANNEL), [SEQUENCE](\ref
   * #ImageDimension.SEQUENCE)) */
  static const ImageType MULTISPECTRAL_VOLUME_SEQUENCE;
};

// create an alias for ImageTypeId
using AxesInterpretationId = ImageTypeId;

} // end namespace iolink
