// ================================================================================ //
//       THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),        //
//                  AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                   //
//                                                                                  //
//     REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS      //
//     SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR      //
//                       WRITTEN AUTHORIZATION OF FEI S.A.S.                        //
//                                                                                  //
//                             RESTRICTED RIGHTS LEGEND                             //
//     USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS     //
//     WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN     //
//     SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT     //
//     CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN     //
//        TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.         //
//                                                                                  //
//                         COPYRIGHT (C) 2021 BY FEI S.A.S,                         //
//                                 BORDEAUX, FRANCE                                 //
//                               ALL RIGHTS RESERVED                                //
//                                                                                  //
//        SEE https://developer.openinventor.com/MiscFiles/EULA.pdf FOR MORE        //
// ================================================================================ //

#pragma once

#include <memory>

#include <iolink/IOLinkAPI.h>
#include <iolink/property/ImageInfoProperty.h>
#include <iolink/property/SpatialCalibrationProperty.h>

namespace iolink
{

/**
 * Hold additionnal info for ImageView data interpretation.
 */
class IOLINK_API ImageProperties
{
public:
  /**
   * Create a property set with all values defaulted, according to a data type.
   *
   * @param dataType DataType used to build default ImageProperties
   *
   * @warning In order to build ImageProperties compatible with a specific ImageView,
   * given dataType must be retrieved from this ImageView.
   */
  static std::shared_ptr<ImageProperties> fromDataType(DataType dataType);

  /**
   * Build a ImageProperties from a set of properties.
   *
   * @param calibration SpatialCalibrationProperty to use
   * @param imageInfo ImageInfoProperty to use
   * @return ImageProperties built from given calibration and imageInfo properties
   */
  static std::shared_ptr<ImageProperties> fromProperties(const SpatialCalibrationProperty& calibration,
                                                         const ImageInfoProperty& imageInfo);

  /**
   * Build a ImageProperties by copying properties from another, except the calibration.
   *
   * @param properties original ImageProperties to modify with calibration properties
   * @param calibration SpatialCalibrationProperty to set into given ImageProperties
   * @return new ImageProperties built from given ones and calibration
   */
  static std::shared_ptr<ImageProperties> fromProperties(std::shared_ptr<const ImageProperties> properties,
                                                         const SpatialCalibrationProperty& calibration);

  /**
   * Build a ImageProperties by copying properties from another, except the Image info.
   *
   * @param properties original ImageProperties to modify with ImageInfo properties
   * @param imageInfo ImageInfoProperty to set into given ImageProperties
   * @return new ImageProperties built from given ones and imageInfo
   */
  static std::shared_ptr<ImageProperties> fromProperties(std::shared_ptr<const ImageProperties> properties,
                                                         const ImageInfoProperty& imageInfo);

  /**
   * Build a ImageProperties by directly specifying SpatialCalibration and imageInfo properties.
   * @param calibration SpatialCalibrationProperty to use to initialize ImageProperties
   * @param imageInfo ImageInfoProperty to use to initialize ImageProperties
   */
  ImageProperties(const SpatialCalibrationProperty& calibration, const ImageInfoProperty& imageInfo);

  ImageProperties(const ImageProperties& other);
  ImageProperties& operator=(const ImageProperties& other);

  ImageProperties(ImageProperties&& other) noexcept;
  ImageProperties& operator=(ImageProperties&& other) noexcept;

  /**
   * Return string representation
   */
  std::string toString() const;

  ~ImageProperties();

  /**
   * Create a copy of the properties instance.
   */
  std::shared_ptr<ImageProperties> clone() const;

  /**
   * Return the calibration associated with these properties.
   *
   * This constant variation is provided in order to be called on constant properties.
   */
  const SpatialCalibrationProperty& calibration() const;

  /**
   * Return the image information associated with these properties.
   *
   * This constant variation is provided in order to be called on constant properties.
   */
  const ImageInfoProperty& imageInfo() const;

  /**
   * Return the calibration associated with these properties.
   */
  SpatialCalibrationProperty& calibration();

  /**
   * Return the image information associated with these properties.
   */
  ImageInfoProperty& imageInfo();

  /**
   * Check if the current properties are compatible with given shape and type
   * @param shape shape of an ImageView
   * @param type type of an ImageView
   * @return TRUE if shape and type are compatible with ImageView
   */
  bool isCompatibleWith(const VectorXu64& shape, DataType type) const;

  /**
   * Equality operator
   */
  bool operator==(const ImageProperties& other) const;

private:
  class Private;
  Private* m_private;
};

} // end namespace iolink
