// ================================================================================ //
//       THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),        //
//                  AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                   //
//                                                                                  //
//     REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS      //
//     SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR      //
//                       WRITTEN AUTHORIZATION OF FEI S.A.S.                        //
//                                                                                  //
//                             RESTRICTED RIGHTS LEGEND                             //
//     USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS     //
//     WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN     //
//     SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT     //
//     CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN     //
//        TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.         //
//                                                                                  //
//                         COPYRIGHT (C) 2021 BY FEI S.A.S,                         //
//                                 BORDEAUX, FRANCE                                 //
//                               ALL RIGHTS RESERVED                                //
//                                                                                  //
//        SEE https://developer.openinventor.com/MiscFiles/EULA.pdf FOR MORE        //
// ================================================================================ //

#pragma once

#include <map>
#include <memory>

#include <iolink/IOLinkAPI.h>
#include <iolink/Uri.h>
#include <iolink/storage/HTTPHeaders.h>
#include <iolink/storage/RandomAccess.h>
#include <iolink/storage/StreamAccess.h>

namespace iolink
{

/**
 * A factory to create RandomAccess instances from various ressources.
 */
class IOLINK_API RandomAccessFactory
{
public:
  RandomAccessFactory() = delete;
  RandomAccessFactory(const RandomAccessFactory& other) = delete;            // copy constructor
  RandomAccessFactory(RandomAccessFactory&& other) = delete;                 // move constructor
  RandomAccessFactory& operator=(const RandomAccessFactory& other) = delete; // copy assignment
  RandomAccessFactory& operator=(RandomAccessFactory&& other) = delete;      // move assignment

  /**
   * Create an in-memory RandomAccess of given size.
   */
  static std::shared_ptr<RandomAccess> allocate(size_t size);

  /**
   * Load given RandomAccess in memory.
   * If storage is already an in-memory RandomAccess, return it (no copy are done).
   * @throw InvalidArgument if input randomAccess is null
   */
  static std::shared_ptr<RandomAccess> copyInMemory(std::shared_ptr<RandomAccess> randomAccess);

  /**
   * Get an accessor from a file mapped in memory.
   *
   * @param filePath The path to the file to map.
   * @return A RandomAccess with read and write capabilities.
   *
   * @throw Error If the file does not exist already
   */
  static std::shared_ptr<RandomAccess> mapFile(const std::string& filePath);

  /**
   * Create a RandomAccess from given memory area.
   *
   * @warning The return DataAccess does not have the ownership of given buffer.
   * You must carefully manage lifetime of each objects.
   */
  static std::shared_ptr<RandomAccess> fromBuffer(void* buffer, size_t size);

  /**
   * Create a RandomAccess from given URI, opening URI in Read Only.
   *
   * If the URI scheme is 'file': creates a ReadRandomAccess using StreamAccessFactory::openFileRead().
   * If the URI scheme is 'http(s)': calls openHTTPRead(), using the GET method and no headers.
   */
  static std::shared_ptr<RandomAccess> openURIRead(const Uri& uri);

  /**
   * Create a RandomAccess from given HTTP URI, opening URI in Read Only.
   * Requested HTTP server must support HTTP range requests.
   *
   * @param uri The HTTP(S) URI to get.
   * @param method The HTTP request method.
   * @param headers The HTTP request headers.
   */
  static std::shared_ptr<RandomAccess> openHTTPRead(const Uri& uri,
                                                    const std::string& method,
                                                    const HTTPHeaders& headers);

  /**
   * Creates an accessor to a subset of a RandomAccess.
   *
   * @param src The accessor to create the subset from
   * @param offset The offset from the start of the source accessor
   *
   * @throw InvalidArgument if input randomAccess is null
   * @throw Error When offset is out of range.
   */
  static std::shared_ptr<RandomAccess> extractRegion(std::shared_ptr<RandomAccess> src, size_t offset);

  /**
   * Creates an accessor to a subset of a RandomAccess.
   *
   * @param src The accessor to create the subset from
   * @param offset The offset from the start of the source accessor
   * @param size The size of the subset.
   *
   * @throw InvalidArgument if input randomAccess is null
   * @throw Error When the subset range is out of the source accessor range.
   */
  static std::shared_ptr<RandomAccess> extractRegion(std::shared_ptr<RandomAccess> src, size_t offset, size_t size);

  /**
   * Create a RandomAccess that will delegate to a StreamAccess.
   * @throw InvalidArgument if input stream is null
   */
  static std::shared_ptr<RandomAccess> fromStream(std::shared_ptr<StreamAccess> stream);
};

} // end namespace iolink