// ================================================================================ //
//       THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),        //
//                  AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                   //
//                                                                                  //
//     REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS      //
//     SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR      //
//                       WRITTEN AUTHORIZATION OF FEI S.A.S.                        //
//                                                                                  //
//                             RESTRICTED RIGHTS LEGEND                             //
//     USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS     //
//     WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN     //
//     SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT     //
//     CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN     //
//        TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.         //
//                                                                                  //
//                         COPYRIGHT (C) 2021 BY FEI S.A.S,                         //
//                                 BORDEAUX, FRANCE                                 //
//                               ALL RIGHTS RESERVED                                //
//                                                                                  //
//        SEE https://developer.openinventor.com/MiscFiles/EULA.pdf FOR MORE        //
// ================================================================================ //

#pragma once

#include <memory>
#include <string>

#include <iolink/IOLinkAPI.h>
#include <iolink/view/Extension.h>

namespace iolink
{

/**
 * Extension to handle units of columns in a dataframe.
 *
 * This extension allows to get and set the unit of a column in a dataframe.
 *
 * The unit is an UTF-8 encoded string that describes the unit of the column,
 * in a format described as follows:
 *
 * - A scalar unit is an string of alphabetical characters, describing the
 *   base unit. For example, "m" for meters, "s" for seconds, or "Å" for angstroms.
 * - Such scalar can have a dimensional suffix, starting with the `^` character,
 *   followed by an integer number. this number may be prefixed by a sign, `+` or `-`.
 * - Unsuffixed units of unsuffixed units may be combined using the `/` or `.`
 *   character, to form a compound unit. For example, "m/s" for meters per second.
 * - An empty unit, or "" is a valid unit, and is equivalent to a dimensionless unit.
 *
 * Examples of such units:
 *  - "m/s" or "m.s^-1" for meters per second.
 *  - "µm^2" for square micrometers.
 *  - "Ω.s^-1" for ohms per seconds, or more simply as siemens "S".
 */
class IOLINK_API ColumnUnitExtension : public Extension
{
public:
  /**
   * Identifier used to load the extension using the View::extension method.
   */
  static constexpr size_t EXTENSION_ID = 0x4;

  /**
   * @brief Upcast a generic extension to this specific type.
   *
   * @param extension The extension to cast.
   * @return The upcasted extension. This will be null if no upcasting is possible.
   */
  inline static std::shared_ptr<ColumnUnitExtension> cast(const std::shared_ptr<Extension>& extension)
  {
    return std::dynamic_pointer_cast<ColumnUnitExtension>(extension);
  }

  /**
   * @brief Return the unit of the column.
   *
   * @param columnIndex The index of the column.
   * @return The unit of the column.
   *
   * @throw Error if the column index is out of range.
   */
  virtual const std::string& unit(size_t columnIndex) const = 0;

  /**
   * @brief Set the unit of the column.
   *
   * @param columnIndex The index of the column.
   * @param unit The unit of the column.
   *
   * @throw Error if the column index is out of range.
   * @throw Error if DataFrameView does not have the WRITE capability.
   */
  virtual void setUnit(size_t columnIndex, const std::string& unit) = 0;
};

} // namespace iolink
