/*=======================================================================
 ***         THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S,                  ***
 ***                   A PART OF THERMO FISHER SCIENTIFIC,                          ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC.       ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***   COPYRIGHT (C) 2021-2025 BY FEI S.A.S, A PART OF THERMO FISHER SCIENTIFIC,    ***
 ***                       BORDEAUX, FRANCE                                         ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
#pragma once

#include <ImageDev/ImageDevCppExports.h>
#include <memory>

namespace imagedev {

class MorphologicalKernel2dPrivate;

/**
 * This class represents a 2D morphological Kernel.
 *
 * A 2D morphological kernel contains exactly 9 values and indexes matching the following table:
 *    4   3   2
 *    5   0   1
 *    6   7   8
 *
 * Accepted values for the kernel are Value:
 * - ZERO: pixel is off
 * - ONE: pixel is on
 * - ANY: pixel is ignored
 */
class IMAGEDEV_CPP_API MorphologicalKernel2d
{
public:
    // possible values
    enum Value
    {
        ZERO = 0,
        ONE = 1,
        ANY = 2
    };

    typedef MorphologicalKernel2d Self;
    typedef std::shared_ptr< Self > Ptr;

    /** constructor returning a shared ptr with all kernel's element initialized with the same value
     * @param value common value for all the kernel's elements.
     */
    static Ptr New( Value value = ZERO );

    /** Get a value of a specific kernel's element
     * @param index position of the kernel's element.
     */
    Value getValue( int index );

    /** Affect a value to a specific kernel's element
     * @param index position of the kernel's element.
     * @param value value to affect.
     */
    void setValue( int index, Value value );

    /** Affect all kernel's elements with the same value
     * @param value default value for all kernel's elements.
     */
    void setAllValues( Value value = ZERO );

    /** Direct rotation of 2 PI / 8
     *    4   3   2         3   2   1
     *    5   0   1   =>    4   0   8
     *    6   7   8         5   6   7
     * @param rotation number of rotation of 2 PI / 8.
     */
    void rotate( size_t rotation = 1 );

    // destructor
    ~MorphologicalKernel2d();

    friend class MorphologicalKernel2dPrivate;
    friend class MorphologicalLut2dPrivate;

protected:
    MorphologicalKernel2d( Value value = ZERO );

    MorphologicalKernel2dPrivate* m_p;
};

} // namespace imagedev
